/*
 * client/DungeonScreen.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "DungeonScreen.h"
#include "ClientLightmap.h"
#include "ClientSector.h"
#include "ClientSide.h"
#include "ClientSound.h"
#include "ClientSprite.h"

#include "../server/Player.h"
#include "../server/Zone.h"
#include "../server/LineDef.h"
#include "../server/SideDef.h"
#include "../server/Sector.h"
#include "../server/SectorSpecial.h"

#include "../server/Sound.h"
#include "../server/Lightmap.h"
#include "../server/Projectile.h"
#include "../server/Sprite.h"

#include "../widgets/WidgetManager.h"
#include "../util/TextureManager.h"
#include "../util/ModelManager.h"
#include "../util/VMath.h"

#include "../util/md2_model.h"

#include "../gl_stubs.h"
#include "../GraphicsState.h"

#include <signal.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL/SDL_mixer.h>

#ifndef M_PI
#define M_PI           3.14159265358979323846  /* pi */
#endif


TextureObject * weapon_textures[10];
md2_model_t * weapon_models[10];
int cur_weapon = 1;

/* in degrees */
float camera_angle_z = 0;//45;
/* in degrees */
float camera_angle_x = -90;//0;//-60;
int zoom = 50;//600;
int fov  = 90;
float last_pos[3] = { 0, 0, 0 };

Mix_Chunk * footstep_wavs[4] = { 0, 0, 0, 0 };
Mix_Chunk * turn_wav = 0;
Mix_Chunk * pain_fall_wav = 0;
Mix_Chunk * pain2_wav = 0;
Mix_Chunk * explode_wav = 0;
Mix_Chunk * fire_wav = 0;
Mix_Chunk * slash_wav = 0;
Mix_Chunk * jump_wav = 0;

enum CameraMode {
        FIRST_PERSON,
        THIRD_PERSON_FOLLOW,
        THIRD_PERSON
};

CameraMode _cameraMode = FIRST_PERSON;//THIRD_PERSON;

int headbob = 0;
int headbob_delta = 3;
int headbob_max   = 5;
int headbob_cur   = 0;
int cur_footstep  = 0;
int num_footsteps = 4;
int footstep_map[] = { 0, 1, 2, 3,
                       1, 0, 3, 2,
                       0, 2, 3, 1 };
int num_mapped_footsteps = 12;
        
static bool do_fill = true;// false;
static bool draw_sides = true;//
//static bool draw_ceilings = true;//false;
static bool draw_textures = true;
static bool draw_lightmaps = true;
static bool draw_invisible_sectors = false;
static bool draw_using_pvs = true;
static bool draw_map_overlay = false;
static bool draw_bounding = false;
static bool draw_weapon = false;//true;
static const int MAP_ZOOM_BASE = 16;//
static int  map_zoom_factor = MAP_ZOOM_BASE/2;//

static GLuint dummy_texture = 0;
static GLuint automap_playerdisc_list = 0;

/** scale of 1 to 10 */
static int viewport_scale = 10;
        

/**
 * Setup the texture and texture coord pointer for this wall section
 *
 * @param side the SideDef to be drawn
 * @param which the section of the side to draw [0-2]
 * @param textype true for texture false for lightmap
 */

void setupWallSurfaceTexture( const class SideDef * side,
                              int which,
                              bool textype ) {

        // normal texture
        
        if ( textype ) {
                glBindTexture( GL_TEXTURE_2D, 
                               side->clientSide->textures[ which ] ?
                               side->clientSide->textures[ which ]->getIndex() : 0 );
                glTexCoordPointer( 2, GL_FLOAT, 0, side->clientSide->texVerts[ which ] );
        }
        
        // lightmap

        else {
                class Lightmap * lm = side->lightmaps[ which ];
                
                if ( lm && lm->clientLightmap ) {

                        if ( lm->needs_rebind )
                                lm->clientLightmap->rebindTexture( lm );

                        glBindTexture( GL_TEXTURE_2D, lm->clientLightmap->_tex );
                        glTexCoordPointer( 2, GL_FLOAT, 0, side->clientSide->lmVerts[which] );
                }
                else {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                }
        }
}

/**
 * Draw all visible sections of the given side (up to 3 sections), using a
 * single texturing pass.
 * 
 * @param side the SideDef to be drawn
 * @param textype true if texture, false if lightmap
 */

static void drawSideSingleTex( const class SideDef * side,
                               bool textype  ) {
        
        glColor3f( 1, 1, 1 );
        
        for ( int i = 0; i < 3; ++i ) {
                
                if ( side->clientSide->verts[ i ] == 0 )
                        continue;
                
                setupWallSurfaceTexture( side, i, textype );
                glVertexPointer( 3, GL_FLOAT, 0, side->clientSide->verts[ i ] );
                glDrawArrays( GL_QUADS, 0, 4 );

                ++g_state.num_surfs;
        }
}

/**
 * Draw all visible sections of the given side (up to 3 sections), using a
 * single MULTItexture pass.  The multitexture pass draws lightmap and texture,
 * if enabled.
 * 
 * @param side the SideDef to be drawn
 */

static void drawSideMultiTex( const class SideDef * side ) {

        glColor3f( 1, 1, 1 );
        
        for ( int i = 0; i < 3; ++i ) {
                
                if ( side->clientSide->verts[i] == 0 )
                        continue;
                
                if ( draw_textures ) {
                        glActiveTextureARB_ptr( GL_TEXTURE0_ARB );
                
                        glBindTexture( GL_TEXTURE_2D, 
                                       side->clientSide->textures[ i ] ?
                                       side->clientSide->textures[ i ]->getIndex() : 0 );
                        
                        glClientActiveTextureARB_ptr( GL_TEXTURE0_ARB );
                        glTexCoordPointer( 2, GL_FLOAT, 0, side->clientSide->texVerts[ i ] );
                        glEnableClientState( GL_TEXTURE_COORD_ARRAY );
                        ++g_state.num_surfs;
                }
                
                class Lightmap * lm = side->lightmaps[i];

                if ( lm && lm->clientLightmap ) {

                        glActiveTextureARB_ptr( GL_TEXTURE1_ARB );

                        if ( lm->needs_rebind )
                                lm->clientLightmap->rebindTexture( lm );

                        glBindTexture( GL_TEXTURE_2D, lm->clientLightmap->_tex );
                        glClientActiveTextureARB_ptr( GL_TEXTURE1_ARB );
                        glTexCoordPointer( 2, GL_FLOAT, 0, side->clientSide->lmVerts[ i ] );
                        glEnableClientState( GL_TEXTURE_COORD_ARRAY );
                        ++g_state.num_surfs;
                }
                
                glVertexPointer( 3, GL_FLOAT, 0, side->clientSide->verts[ i ] );
                glEnable( GL_VERTEX_ARRAY );
                
                glDrawArrays( GL_QUADS, 0, 4 );
                
        }
}

/**
 * Setup the texture parameters for drawing a single-textured floor or ceiling surface
 * @param sect the sector to draw
 * @param which floor (0) or ceiling (1)
 * @param textype setup the texture (true) or lightmap( false)
 */

void setupHorizontalSurfaceTexture( const class Sector * sect,
                                    Sector::SurfaceType which,
                                    bool textype ) {

        class ClientSector * cs = sect->clientSector;

        if ( textype ) {
                glBindTexture( GL_TEXTURE_2D,
                               cs->to[ which ] ? 
                               cs->to[ which ]->getIndex() : 0 );
                glTexCoordPointer( 2, GL_FLOAT, 0, cs->textureVerts[ which ] );
        }
        else {

                // fullbright textures start with '_' for now
                // FIXME: make a fullbright flag for textures
                
                if ( sect->getTex( which ) && sect->getTex( which )[9] == '_' ) {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                        return;
                }

                class Lightmap * lm = which == 0 ? sect->floor_lightmap : sect->ceiling_lightmap;
                
                if ( lm ) {
                        if ( lm->needs_rebind )
                                lm->clientLightmap->rebindTexture( lm );

                        glBindTexture( GL_TEXTURE_2D, lm ? lm->clientLightmap->_tex : 0 );
                        
                        glTexCoordPointer( 2, GL_FLOAT, 0, cs->lmVerts );
                }   
                else {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                }
        }
}

/**
 * Setup the texture parameters for drawing a multi-textures floor or ceiling surface
 * The lightmap and texture are blended together.
 *
 * @param sect the sector to draw
 * @param which floor (0) or ceiling (1)
 */

void setupHorizontalSurfaceMultiTexture( const class Sector * sect, 
                                         Sector::SurfaceType which ) {

        class ClientSector * cs = sect->clientSector;

        glActiveTextureARB_ptr( GL_TEXTURE0_ARB );
        
        if ( draw_textures ) {
                glBindTexture( GL_TEXTURE_2D,
                               cs->to[ which ] ? 
                               cs->to[ which ]->getIndex() : 0 );
                glClientActiveTextureARB_ptr( GL_TEXTURE0_ARB );
                glTexCoordPointer( 2, GL_FLOAT, 0, cs->textureVerts[ which ] );
                //                glEnable( GL_TEXTURE_COORD_ARRAY );
                glEnableClientState(GL_TEXTURE_COORD_ARRAY);
        }
        
        if ( draw_lightmaps ) {
                glActiveTextureARB_ptr( GL_TEXTURE1_ARB );
                
                // fullbright textures start with '_' for now
                // FIXME: make a fullbright flag for textures
                
                if ( sect->getTex( which ) && sect->getTex( which )[9] == '_' ) {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                        return;
                }

                class Lightmap * lm = which == 0 ? sect->floor_lightmap : sect->ceiling_lightmap;
        
                if ( lm ) {
                        
                        if ( lm->needs_rebind )
                                lm->clientLightmap->rebindTexture( lm );

                        glBindTexture( GL_TEXTURE_2D, lm ? lm->clientLightmap->_tex : 0 );

                        glClientActiveTextureARB_ptr( GL_TEXTURE1_ARB );
                        glTexCoordPointer( 2, GL_FLOAT, 0, cs->lmVerts );
                        glEnableClientState(GL_TEXTURE_COORD_ARRAY);
                }
                
                else {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                }
                
        }
        
     
}

/**
 * Draw a horizontal surface.  This is called after any texturing setup is done.
 *
 */

void drawHorizontalSurface( const class Sector * sect, 
                            int which ) {
        
        class ClientSector * cs = sect->clientSector;
        
        glColor3f( 1, 1, 1 );
        
        glVertexPointer( 2, GL_FLOAT, 0, cs->verts );
        glEnable( GL_VERTEX_ARRAY );
        glMatrixMode( GL_MODELVIEW );
        glPushMatrix();
        
        if ( which == 0 ) {
                

                glTranslatef( 0, 0, sect->getFloorHeight() );
                
                //
                // floors are wound backwards?
                //
                
                glFrontFace( GL_CW );
                glDrawArrays( GL_TRIANGLE_FAN, 0, sect->verts.length );
                glFrontFace( GL_CCW );
        }
        else {
                glTranslatef(  0, 0, sect->getCeilingHeight() );
                glDrawArrays( GL_TRIANGLE_FAN, 0, sect->verts.length );
        }

        glDisable( GL_VERTEX_ARRAY );
        glPopMatrix();
        
}

/**
 * 
 */
static void renderMultiPassWallSurfaces( const class Sector * pSect,
                                         const Array< class LineDef * > & drawLines,
                                         const Array< class LineDef * > & lines,
                                         const GLenum depth_type,
                                         const bool textype ) {
        
        glDepthFunc( depth_type );
        
        for ( int i = 0; i < drawLines.length; ++i ) {
                        
                class LineDef * line = drawLines[ i ];
                        
                drawSideSingleTex( line->sides[0], textype );
                        
                if ( line->sides[1] ) {
                        drawSideSingleTex( line->sides[1], textype );
                }
        }
                
        if ( pSect ) {
                //
                // now draw the walls of the current sector with depth test disabled
                //
                
                glDepthFunc( GL_ALWAYS );
                
                for ( unsigned int i = 0; i < pSect->_myLines.size(); ++i ) {
                        class LineDef * line = lines[ pSect->_myLines[i] ];
                        
                        if ( line->modified ) {
                                class SideDef ** sides = line->sides;
                                
                                sides[0]->clientSide->adjustHeights( sides[0], sides[1] );
                                if ( sides[1] )
                                        sides[1]->clientSide->adjustHeights( sides[1], sides[0] );
                                line->modified = false;
                        }
                        
                        if ( line->sides[0]->sector == pSect ) {
                                drawSideSingleTex( line->sides[0], textype );
                        }
                        else {
                                drawSideSingleTex( line->sides[1], textype );
                        }
                }
        }
}

static void renderMultiPassSectorSurfaces( const class Sector * pSect,
                                           const Array<Sector *> & sectors,
                                           const float eye_z,
                                           const bool textype,
                                           const bool handle_specials ) {
        

        vector<const class Sector *> specialFloors;
        vector<const class Sector *> specialCeilings;
        
        for ( unsigned int i = 0; i < pSect->_visibleFloors.size(); ++i ) {
                class Sector * vsect = sectors[ pSect->_visibleFloors[i] ];
                if ( eye_z > vsect->getFloorHeight() ) {
    
                        if ( handle_specials && vsect->spec && vsect->spec->isLift() ) {
                                specialFloors.push_back( vsect );
                                continue;
                        }
                        
                        setupHorizontalSurfaceTexture( vsect, Sector::FLOOR, textype );
                        drawHorizontalSurface( vsect, 0 );
                }
        }

        for ( unsigned int i = 0; i < pSect->_visibleCeilings.size(); ++i ) {
                class Sector * vsect = sectors[ pSect->_visibleCeilings[i] ];
                if ( eye_z < vsect->getCeilingHeight() ) {

                        if (  handle_specials && vsect->spec && vsect->spec->isDoor() ) {
                                specialCeilings.push_back( vsect );
                                continue;
                        }

                        setupHorizontalSurfaceTexture( vsect, Sector::CEILING, textype );
                        drawHorizontalSurface( vsect, 1 );
                }
        }
                        
        setupHorizontalSurfaceTexture( pSect, Sector::FLOOR, textype );
        drawHorizontalSurface( pSect, 0 );
        setupHorizontalSurfaceTexture( pSect, Sector::CEILING, textype );
        drawHorizontalSurface( pSect, 1 );

        if ( handle_specials ) {
                glDepthFunc( GL_LESS );
                        
                if ( specialFloors.size() ) {
                        for ( unsigned int i = 0; i < specialFloors.size(); ++i ) {
                                setupHorizontalSurfaceTexture( specialFloors[i], Sector::FLOOR, textype );
                                drawHorizontalSurface( specialFloors[i], 0 ); 
                        }
                }

                if ( specialCeilings.size() ) {
                        for ( unsigned int i = 0; i < specialCeilings.size(); ++i ) {
                                setupHorizontalSurfaceTexture( specialCeilings[i], Sector::CEILING, textype );
                                drawHorizontalSurface( specialCeilings[i], 1 ); 
                        }
                }
        }
}
                                           

static void drawWalls( const class Zone * zone,
                       const class Player * player,
                       const class Sector * pSect ) {
        
        const Array<class LineDef *> & lines = zone->getLines();
        
        if ( ! draw_using_pvs || ! zone->isVisLoaded() )
                pSect = 0;
        
        Array< class LineDef * > drawLines( pSect ? 
                                            pSect->_visibleLines.size() :
                                            ( lines.length ) );
        
        if ( pSect ) {
                for ( unsigned int i = 0; i < pSect->_visibleLines.size(); ++i ) {
                        drawLines[i] = lines[ pSect->_visibleLines[ i ] ];

                        if ( drawLines[i]->modified ) {
                                class SideDef ** sides = drawLines[i]->sides;
                                
                                sides[0]->clientSide->adjustHeights( sides[0], sides[1] );
                                if ( sides[1] )
                                        sides[1]->clientSide->adjustHeights( sides[1], sides[0] );
                                drawLines[i]->modified = false;
                        }
                }
        }

        else {
                for ( int j = 0; j < lines.length; ++j ) {
                        drawLines[ j ] = lines[ j ];
                        
                        if ( drawLines[j]->modified ) {
                                class SideDef ** sides = drawLines[j]->sides;
                                
                                sides[0]->clientSide->adjustHeights( sides[0], sides[1] );
                                if ( sides[1] )
                                        sides[1]->clientSide->adjustHeights( sides[1], sides[0] );
                                drawLines[j]->modified = false;
                        }
                }
        }
          
        //
        //
        //

        if ( g_state.use_multitex ) {
                glEnable( GL_DEPTH_TEST );
                glDepthFunc( GL_LESS );
                
                for ( int i = 0; i < drawLines.length; ++i ) {
                        
                        class LineDef * line = drawLines[ i ];
                        
                        drawSideMultiTex( line->sides[0] );
                        
                        if ( line->sides[1] ) {
                                drawSideMultiTex( line->sides[1] );
                        }
                }
                
                if ( pSect ) {
                        //
                        // now draw the walls of the current sector with depth test disabled
                        //
                
                        glDepthFunc( GL_ALWAYS );
                
                        for ( unsigned int i = 0; i < pSect->_myLines.size(); ++i ) {
                                class LineDef * line = lines[ pSect->_myLines[i] ];
                                
                                if ( line->modified ) {
                                        class SideDef ** sides = line->sides;
                                        
                                        sides[0]->clientSide->adjustHeights( sides[0], sides[1] );
                                        if ( sides[1] )
                                                sides[1]->clientSide->adjustHeights( sides[1], sides[0] );
                                        line->modified = false;
                                }

                                if ( line->sides[0]->sector == pSect ) {
                                        drawSideMultiTex( line->sides[0] );
                                }
                                else {
                                        drawSideMultiTex( line->sides[1] );
                                }
                        }

                        glDepthFunc( GL_LESS );
                }
        }
        
        //
        // use multipass
        //

        else {
                
                //
                // draw all the potentially visible walls
                //
                glEnable( GL_TEXTURE_2D );
                glEnable( GL_TEXTURE_COORD_ARRAY );
                glEnable( GL_VERTEX_ARRAY );
                
                glEnable( GL_DEPTH_TEST );
                glDepthFunc( GL_LESS );
                
                if ( draw_lightmaps ) {
                        renderMultiPassWallSurfaces( pSect, drawLines, lines, GL_LESS, false );

                        if ( draw_textures ) {
                                glDepthMask( GL_FALSE );
                                        
                                glEnable( GL_BLEND );
                                glBlendFunc( GL_ZERO, GL_SRC_COLOR );

                                renderMultiPassWallSurfaces( pSect, drawLines, lines, GL_EQUAL, true );

                                glDepthMask( GL_TRUE );
                                glDisable( GL_BLEND );

                        }
                }
                else if ( draw_textures ) {
                        renderMultiPassWallSurfaces( pSect, drawLines, lines, GL_LESS, true );
                }
                        
                glDepthFunc( GL_LESS );
        }
}


static void drawSectors( const class Zone * zone,
                         const class Player * player,
                         const class Sector * pSect ) {
        //
        // TODO: draw only sector floors which are below the eyepoint
        // TODO: draw only sector ceilings which are above the eyepoint
        // TODO: draw the sector floor/ceilings with depth test disabled
        //

        float eye_z = player->getPos()[2] + player->getEyeLevel();
        
        const Array<Sector *> & sectors = zone->getSectors();

        if ( pSect && draw_using_pvs && zone->isVisLoaded() ) {

                g_state.num_surfs += pSect->_visibleFloors.size() +  pSect->_visibleCeilings.size() + 2;
                
                //
                // use multitexture rendering
                //

                if ( g_state.use_multitex ) {

                        vector<const class Sector *> specialFloors;
                        vector<const class Sector *> specialCeilings;
                
                       glEnable( GL_DEPTH_TEST );
                        glDepthFunc( GL_ALWAYS );

                        for ( unsigned int i = 0; i < pSect->_visibleFloors.size(); ++i ) {
                                class Sector * vsect = sectors[ pSect->_visibleFloors[i] ];
                                if ( eye_z > vsect->getFloorHeight() ) {
                                        if ( vsect->spec && vsect->spec->isLift() ) {
                                                specialFloors.push_back( vsect );
                                                continue;
                                        }
                                        
                                        setupHorizontalSurfaceMultiTexture( vsect, Sector::FLOOR );
                                        drawHorizontalSurface( vsect, 0 );
                                }
                        }

                        for ( unsigned int i = 0; i < pSect->_visibleCeilings.size(); ++i ) {
                                class Sector * vsect = sectors[ pSect->_visibleCeilings[i] ];
                                if ( eye_z < vsect->getCeilingHeight() ) {

                                        if ( vsect->spec && vsect->spec->isDoor() ) {
                                                specialCeilings.push_back( vsect );
                                                continue;
                                        }
                                        
                                        setupHorizontalSurfaceMultiTexture( vsect, Sector::CEILING );
                                        drawHorizontalSurface( vsect, 1 );
                                }
                        }
                        
                        setupHorizontalSurfaceMultiTexture( pSect, Sector::FLOOR );
                        drawHorizontalSurface( pSect, 0 );
                        setupHorizontalSurfaceMultiTexture( pSect, Sector::CEILING );
                        drawHorizontalSurface( pSect, 1 );

                        glDepthFunc( GL_LESS );
                        
                        if ( specialFloors.size() ) {
                                for ( unsigned int i = 0; i < specialFloors.size(); ++i ) {
                                       setupHorizontalSurfaceMultiTexture( specialFloors[i], Sector::FLOOR );
                                       drawHorizontalSurface( specialFloors[i], 0 ); 
                                }
                        }

                        if ( specialCeilings.size() ) {
                                for ( unsigned int i = 0; i < specialCeilings.size(); ++i ) {
                                       setupHorizontalSurfaceMultiTexture( specialCeilings[i], Sector::CEILING );
                                       drawHorizontalSurface( specialCeilings[i], 1 ); 
                                }
                        }
                        
                }
                
                //
                // use multipass rendering
                //

                else {

                        glEnable( GL_TEXTURE_2D );
                        glEnable( GL_TEXTURE_COORD_ARRAY );

                        glEnable( GL_DEPTH_TEST );
                        glDepthFunc( GL_ALWAYS );

                        if ( draw_lightmaps ) {
                                renderMultiPassSectorSurfaces( pSect, sectors, eye_z, false, true );

                                if ( draw_textures ) {

                                        glDepthFunc( GL_EQUAL );
                                        glDepthMask( GL_FALSE );
                                        
                                        glEnable( GL_BLEND );
                                        glBlendFunc( GL_ZERO, GL_SRC_COLOR );

                                        renderMultiPassSectorSurfaces( pSect, sectors, eye_z, true, false );

                                        glDepthMask( GL_TRUE );
                                        glDisable( GL_BLEND );
                                }
                        }
                        else if ( draw_textures ) {
                                renderMultiPassSectorSurfaces( pSect, sectors, eye_z, true, true );

                               
                        }
                        
                        glDepthFunc( GL_LESS );
                        glEnable( GL_DEPTH_TEST );
                }
                return;
        }

        //
        // simply draw all sectors
        //

        g_state.num_surfs += sectors.length * 2;

        if ( g_state.use_multitex ) {
                for ( int i = 0; i < sectors.length; ++i ) {
                        setupHorizontalSurfaceMultiTexture( sectors[i], Sector::FLOOR );
                        drawHorizontalSurface( sectors[i], 0 );
                        setupHorizontalSurfaceMultiTexture( sectors[i], Sector::CEILING );
                        drawHorizontalSurface( sectors[i], 1 );
                }
        }
        else {
                for ( int i = 0; i < sectors.length; ++i ) {
                        drawHorizontalSurface( sectors[i], 0 );
                        drawHorizontalSurface( sectors[i], 1 );
                }
        }

}

static void drawZoneMap( const class Player * player ) {

        glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

        const class Zone * zone = player->getContainerZone();
                
        const Array<Sector *> & sectors = zone->getSectors();
        
        const float sect_color[4] = { 1, 1, 1, 0.1 };

        glColor4fv( sect_color );
        
        for ( int i = 0; i < sectors.length; ++i ) { 

                if ( sectors[i]->spec )
                        glColor4f( 1, 0.5, 1, 0.2 );
                
                glBegin( GL_POLYGON );
                {
                        for ( int j = sectors[i]->verts.length-1; j >= 0; --j ) {
                                glVertex3f( sectors[i]->verts[j][0], 
                                            sectors[i]->verts[j][1], 0 );
                        }
                }
                glEnd();

                if ( sectors[i]->spec )
                        glColor4fv( sect_color );

        }
        const Array<class LineDef *> & lines = zone->getLines();

        glDisable( GL_BLEND );

        glBegin( GL_LINES );
        {
                for ( int i = 0; i < lines.length; ++i ) {
                        // 2-sided line
                        if ( lines[i]->sides[1] ) {
                                if ( lines[i]->sides[0]->sector->getFloorHeight() ==
                                     lines[i]->sides[1]->sector->getFloorHeight() ) {
                                        continue;
                                }
                                glColor3f( 0.5, 0.5, 0.5 );
                        }
                        else {
                                glColor3f( 1, 1, 1 );
                        }
                
                        glVertex2fv( lines[i]->verts[0] );
                        glVertex2fv( lines[i]->verts[1] );
                }
        }
        glEnd();

}

static void drawZone( const class Player * player ) {
                    
        const class Zone * zone = player->getContainerZone();
        
        g_state.num_surfs = 0;
        g_state.num_ents = 0;

        if ( !do_fill )
                glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

        if ( g_state.use_multitex ) {
                glActiveTextureARB_ptr( GL_TEXTURE0_ARB );

                if ( draw_textures ) {
                        glBindTexture( GL_TEXTURE_2D, 0 );
                }
                else {
                        glBindTexture( GL_TEXTURE_2D, dummy_texture );
                }
        
                glEnable( GL_TEXTURE_2D );
        
                glActiveTextureARB_ptr( GL_TEXTURE1_ARB );
                glBindTexture( GL_TEXTURE_2D, 0 );

                if ( draw_lightmaps ) {
                        glEnable( GL_TEXTURE_2D );
                
                }
                else {
                        glDisable( GL_TEXTURE_2D ); 
                }
        }
        else {
                glBindTexture( GL_TEXTURE_2D, 0 );
        }
        
        const class Sector * pSect = player->getSector();
        
        drawSectors( zone, player, pSect );
        
        if ( draw_sides )
                drawWalls( zone, player, pSect );
        
        if ( g_state.use_multitex ) {
                glActiveTextureARB_ptr( GL_TEXTURE1_ARB );
                glBindTexture( GL_TEXTURE_2D, dummy_texture );
                glDisable( GL_TEXTURE_2D );
                
                glActiveTextureARB_ptr( GL_TEXTURE0_ARB );
                glBindTexture( GL_TEXTURE_2D, dummy_texture );
                glDisable( GL_TEXTURE_2D );
        }
        
        else {
                glDisable( GL_TEXTURE_2D );
                glDisable( GL_TEXTURE_COORD_ARRAY );
        }
}

static void drawDisc( const float radius, float r, float g, float b ) {
        
        const int NUM_VERTS = 9;
        
        float verts[ NUM_VERTS + 1 ][2]; 
        double angle = 0;

        for ( int i = 0; i < NUM_VERTS; ++i ) {
                angle = i * M_PI * 2 / NUM_VERTS;
                verts[i][0] = cos( angle ) * radius;
                verts[i][1] = -sin( angle ) * radius;
        }

        verts[ NUM_VERTS ][0] = verts[0][0];
        verts[ NUM_VERTS ][1] = verts[0][1];
        
        glBegin( GL_TRIANGLE_FAN );
        {
                glColor3f( r/2, g/2, b/2 );
                
                glVertex2f( 0, 0 );

                glColor3f( r, g, b );

                for ( int i = NUM_VERTS; i >=0; --i ) {
                        glVertex2f( verts[i][0], verts[i][1] );
                }
        }
        glEnd();

}

static void drawBounding( const class Entity * ent, bool reverse, float r, float g, float b ) {

        int radius = ent->getBoundingRadius();
        int height = ent->getHeight();
        
        const int MAX_NUM_VERTS = 16;
        
        int NUM_VERTS = 16;
        
        float verts[ MAX_NUM_VERTS + 1 ][2];
        double angle = 0;
        
        if ( reverse ) {
                for ( int i = 0; i < NUM_VERTS; ++i ) {
                        angle = i * M_PI * 2 / NUM_VERTS;
                        verts[i][0] = cos( angle ) * radius;
                        verts[i][1] = sin( angle ) * radius;
                }
        }
        else {
                for ( int i = 0; i < NUM_VERTS; ++i ) {
                        angle = i * M_PI * 2 / NUM_VERTS;
                        verts[i][0] = cos( angle ) * radius;
                        verts[i][1] = -sin( angle ) * radius;
                }

        }

        verts[ NUM_VERTS ][0] = verts[0][0];
        verts[ NUM_VERTS ][1] = verts[0][1];
                
        float dark[3] = {
                r / 2,
                g / 2,
                b / 2
        };
        
        glColor3f( r, g, b );

        //        if ( !do_fill )

        glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

        glBegin( GL_QUAD_STRIP );
        {
                for ( int i = 0; i <= NUM_VERTS ; ++i ) {
                        
                        glVertex3f( verts[i][0], verts[i][1], 0 );
                        glVertex3f( verts[i][0], verts[i][1], height );
                }
        }
        glEnd();

        //
        // top
        //


        glColor3fv( dark );

        glBegin( GL_TRIANGLE_FAN );
        {
                glVertex3f( 0, 0, height );

                for ( int i = NUM_VERTS; i >=0; --i ) {
                        glVertex3f( verts[i][0], verts[i][1], height );
                }
        }
        glEnd();

        //
        // bottom
        //
        
        glBegin( GL_TRIANGLE_FAN );
        {
                glVertex3f( 0, 0, 0 );

                for ( int i = 0; i <= NUM_VERTS; ++i ) {
                        glVertex3f( verts[i][0], verts[i][1], 0 );
                }
        }
        glEnd();
        
        float theta = VMath::degrees2radians( ent->getTilt() );

        glColor3f( 1, 0, 1 );
        glBegin( GL_LINES );
        {
                glVertex3f( 0, 0, height / 2 );
                glVertex3f( radius + 50 , 0, height / 2 + ( radius + 50 ) * sin( theta ) );
        }
        glEnd();

        if ( do_fill )
                glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

}


static GLuint generateBounding( const class Entity * ent, bool reverse, float r, float g, float b ) {

        GLuint alist = glGenLists( 1 );
           
        if ( alist == 0 ) {
                cout << "ERROR" << endl;
                return 0;
        }

        glNewList( alist, GL_COMPILE );
        {
                drawBounding( ent, reverse, r, g, b );
        }
        glEndList();

        return alist;
}


DungeonScreen::DungeonScreen( int w, int h, Client::State escape, Client::State close )
        : GenericScreen( w, h, escape, close ),
          _playerList(0)
{
        _name = "DungeonScreen";

        weapon_textures[0] = TextureManager::getTexture( "models/pknight/flamesword", GL_RGB );
        weapon_textures[1] = TextureManager::getTexture( "models/pknight/axe", GL_RGB );
        weapon_textures[2] = TextureManager::getTexture( "models/pknight/battleaxe", GL_RGB );
        weapon_textures[3] = TextureManager::getTexture( "models/pknight/bastard_sword", GL_RGB );
        weapon_textures[4] = TextureManager::getTexture( "models/pknight/split_sword", GL_RGB );
        weapon_textures[5] = TextureManager::getTexture( "models/pknight/morning_star", GL_RGB );
        weapon_textures[6] = TextureManager::getTexture( "models/pknight/spiked_flail", GL_RGB );
        weapon_textures[7] = TextureManager::getTexture( "models/pknight/super_flamesword", GL_RGB );
        weapon_textures[8] = TextureManager::getTexture( "models/pknight/icesword", GL_RGB );
        weapon_textures[9] = TextureManager::getTexture( "models/pknight/lightning_sword", GL_RGB );

        weapon_models[0] = ModelManager::getModel( "pknight/flamesword" );
        weapon_models[1] = ModelManager::getModel( "pknight/axe" );
        weapon_models[2] = ModelManager::getModel( "pknight/battleaxe" );
        weapon_models[3] = ModelManager::getModel( "pknight/bastard_sword" );
        weapon_models[4] = ModelManager::getModel( "pknight/split_sword" );
        weapon_models[5] = ModelManager::getModel( "pknight/morning_star" );
        weapon_models[6] = ModelManager::getModel( "pknight/spiked_flail" );
        weapon_models[7] = ModelManager::getModel( "pknight/super_flamesword" );
        weapon_models[8] = ModelManager::getModel( "pknight/icesword" );
        weapon_models[9] = ModelManager::getModel( "pknight/lightning_sword" );


        setUpdateOk( false );

        //
        // obtain the dummy texture
        // 

        dummy_texture = TextureManager::getDummyTexture();
        
        Client::getWidgetManager()->addUpdateWidget( (class Widget *) this );

        if ( ! footstep_wavs[0] )
                footstep_wavs[0] = Mix_LoadWAV( "lib/sfx/footstep1.wav" );
        if ( ! footstep_wavs[1] )
                footstep_wavs[1] = Mix_LoadWAV( "lib/sfx/footstep2.wav" );
        if ( ! footstep_wavs[2] )
                footstep_wavs[2] = Mix_LoadWAV( "lib/sfx/footstep3.wav" );
        if ( ! footstep_wavs[3] )
                footstep_wavs[3] = Mix_LoadWAV( "lib/sfx/footstep4.wav" );
        
        if ( ! turn_wav )
                turn_wav = Mix_LoadWAV( "lib/sfx/jump_land.wav" );

        if ( ! pain_fall_wav )
                pain_fall_wav = Mix_LoadWAV( "lib/sfx/pain_fall.wav" );

        if ( ! pain2_wav )
                pain2_wav = Mix_LoadWAV( "lib/sfx/pain3.wav" );

        if ( ! explode_wav )
                explode_wav = Mix_LoadWAV( "lib/sfx/big_explode.wav" );

        if ( ! fire_wav )
                fire_wav = Mix_LoadWAV( "lib/sfx/fireloop.wav" );

        if ( ! slash_wav )
                slash_wav = Mix_LoadWAV( "lib/sfx/sword_slash1.wav" );

        if ( ! jump_wav )
                jump_wav = Mix_LoadWAV( "lib/sfx/player_jump1.wav" );


        //------------------------------------------
        /*
        model_lists = glGenLists( model.numFrames );
        for ( int i = 0; i < model.numFrames; ++i ) {
                glNewList( model_lists + i, GL_COMPILE );
                drawModel( i );
                glEndList();
        }
        */
}

void DungeonScreen::resetWorld() {
        
        if ( _playerList != 0 ) {
                glDeleteLists( _playerList, 1 );
                _playerList = 0;
        }

        class Player *player = Client::getPlayer();
        
        cout << " resetWorld with player=" << player << endl;

        if ( ! player )
                return;

        camera_angle_z = VMath::clampAngle360( player->getAngle() );

        //
        // preload textures
        //

        const class Array<const char*> & texnames = player->getContainerZone()->getTexnames();
        
        cout << " ** PRELOADING " << texnames.length  << " textures" << endl;

        for ( int i = 1; i < texnames.length; ++i ) {
                TextureManager::getTexture( texnames[i],   GL_RGB );
        }
        
        //
        // create lightmaps
        //
        
        class Zone * zone = player->getContainerZone();

        const class Array<class Lightmap *> & lms = zone->getLightmaps();
        
        //
        // TODO: make sure client lightmaps are deleted
        //

        for ( int i = 0; i < lms.length; ++i ) {
                ClientLightmap * clm = new ClientLightmap( lms[i] );
                lms[i]->clientLightmap = clm;
        }

        //
        // create sector data
        //

        //
        // TODO: make sure client sector data are deleted
        //

        const Array<Sector *> & sectors = zone->getSectors();

        for ( int i = 0; i < sectors.length; ++i ) {
                ClientSector *cs = new ClientSector( sectors[i] );
                sectors[i]->clientSector = cs;
        }

        const Array<SideDef *> & sides = zone->getSides();

        for ( int i = 0; i < sides.length; ++i ) {
                ClientSide *cs = new ClientSide( sides[i] );
                sides[i]->clientSide = cs;
        }

        _playerList = generateBounding( player, false, 1, 1, 0 );
        
        automap_playerdisc_list   = glGenLists( 1 );

        glNewList( automap_playerdisc_list, GL_COMPILE );
        {
                glShadeModel( GL_FLAT );
                drawDisc( player->getBoundingRadius(), 1, 1, 0 );
        }
        glEndList();
        
        const Vector<class Entity *> & ents = player->getContainerZone()->getContents();
        
        for ( Vector< class Entity * >::const_iterator iter = ents.begin();
              iter != ents.end(); ++iter ) { 
                const class Sprite * sprite = (*iter)->getSprite();
                if ( sprite && ! sprite->clientSprite ) {
                        sprite->clientSprite = new ClientSprite( sprite );
                }
        }
}

DungeonScreen::~DungeonScreen() {
        
        if ( _playerList ) {
                glDeleteLists( _playerList, 1 );
                _playerList = 0;
        }
        
        Client::getWidgetManager()->removeUpdateWidget( (class Widget *) this );
}

static int speed = 10;


void drawModel( int frame_num, md2_model_t & model, const class TextureObject * tex, bool opaque ) {
        
        if ( model.isOk == false )
                return;

        if ( frame_num > model.numFrames ) {
                cout << " :: WARNING: frame " << frame_num << " out of range for " << model << endl;
                frame_num = 0;
        }

        if ( g_state.use_multitex ) {
                glActiveTextureARB_ptr( GL_TEXTURE0_ARB );
        }
        
        glEnable( GL_TEXTURE_2D );
        glBindTexture( GL_TEXTURE_2D, tex->getIndex() );
        
        if ( opaque ) {
                glDisable( GL_BLEND );
                glDisable( GL_ALPHA_TEST );
        }
        else { 
                glEnable( GL_BLEND );
                glEnable( GL_ALPHA_TEST );
                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
        }

        glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
        
        glTexParameterf( GL_TEXTURE_2D,
                         GL_TEXTURE_MIN_FILTER,
                         GL_NEAREST );

        glFrontFace( GL_CW );

        for ( int i = 0; i < model.numGLCommands; ++i ) {
                glBegin( (GLenum) model.gl_cmds[i].gl_type );
                
                float * tc = model.gl_cmds[i].texcoords;
                
                for ( int j = 0; j < model.gl_cmds[i].num_verts; ++j ) {
                        glTexCoord2fv( tc );
                        glVertex3fv(& model.frames[frame_num].verts[ model.gl_cmds[i].verts_i[j] * 3 ] );
                        tc+=2;
                }
                glEnd();
        }

        glDisable( GL_TEXTURE_2D );
        glDisable( GL_BLEND );
        glDisable( GL_ALPHA_TEST );

        glFrontFace( GL_CCW );

}

static void showSprite( const class Character * viewer, 
                        const class Entity * target ) {
        
        const class Sprite * sprite = target->getSprite();
        const class ClientSprite * cs = sprite->clientSprite;
        
        TextureObject * to = cs->texs[ sprite->cur_frame ];
        
        float vect[3] = { 
                target->getPos()[0] - viewer->getPos()[0],
                target->getPos()[1] - viewer->getPos()[1],
                target->getPos()[2] + target->getHeight()/2 - viewer->getPos()[2] - viewer->getEyeLevel()
        };
        
        float angle_z = VMath::radians2degrees( atan2( vect[1], vect[0] ) );
        float dist = VMath::length2d( vect );
        float angle_y = VMath::radians2degrees( atan2( vect[2], dist ) );
        float halfHeight = ((float)target->getHeight())/2;
        
        glTranslatef( 0, 0, target->getHeight()/2 );
        glRotatef( angle_z, 0, 0, 1 );
        glRotatef( angle_y + 90, 0, -1, 0 );
        
        glColor4f( 1, 1, 1, 1 );
        glDepthMask( GL_FALSE );
        glBindTexture( GL_TEXTURE_2D, to->getIndex() );
        glEnable( GL_TEXTURE_2D );
        glEnable( GL_BLEND );
        glEnable( GL_ALPHA_TEST );

        glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

        glBegin( GL_QUADS );
        {
                glTexCoord2d( 0, 0 );
                glVertex2f( -halfHeight, -target->getBoundingRadius() );
                glTexCoord2d( 1, 0 );
                glVertex2f( halfHeight, -target->getBoundingRadius() );
                glTexCoord2d( 1, 1 );
                glVertex2f( halfHeight, target->getBoundingRadius() );
                glTexCoord2d( 0, 1 );
                glVertex2f( -halfHeight, target->getBoundingRadius() );
        }
        glEnd();

        glDisable( GL_BLEND );
        glDisable( GL_ALPHA_TEST );
        glDisable( GL_TEXTURE_2D );

        glDepthMask( GL_TRUE );
}

/**
 * this method assumes that the depth buffer is clear and the color buffer is
 * cleared to the right color, if any
 */

void DungeonScreen::paint() {
        
        GLint viewport[4];
        
        if ( viewport_scale != 10 ) {
                
                glGetIntegerv( GL_VIEWPORT, viewport );
                
                glViewport( viewport[0] + viewport[2] * ( 10 - viewport_scale ) / 20,
                            viewport[1] + viewport[3] * ( 10 - viewport_scale ) / 20,
                            viewport[2] * viewport_scale / 10,
                            viewport[3] * viewport_scale / 10 );
        }
        
        g_state.show_cursor = false;//_cameraMode != FIRST_PERSON;
        
        class Player *player = Client::getPlayer();
        
        glColor3f( 0.0, 0.0, 0.0 );

        //        GLenum gl_error;

        glMatrixMode( GL_PROJECTION );

        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 0 " << (long) gl_error << endl;
        
        glPushMatrix();
        glLoadIdentity();
                
        gluPerspective( fov, 1.3, 4, 5000 );

        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 1 " << (long) gl_error << endl;
                
        glMatrixMode( GL_MODELVIEW );
                
        glLoadIdentity();
     
        if ( _cameraMode == FIRST_PERSON ) {
                glRotatef( player->getTilt() + 90, -1, 0, 0 );
                glRotatef( 90 - player->getAngle(), 0, 0, 1 );
                glTranslatef( -player->getPos()[0], -player->getPos()[1], 
                              -player->getPos()[2] + headbob - player->getEyeLevel() );
                
        }
        else {
                glTranslatef( 0, 0, -zoom );
                glRotatef( camera_angle_x, 1, 0, 0 );
                glRotatef( 90 - camera_angle_z, 0, 0, 1 );
                glTranslatef( -player->getPos()[0], -player->getPos()[1],
                              -player->getPos()[2] - player->getEyeLevel() );
        }

        glEnable( GL_BLEND );

        if ( g_state.polygon_smooth_ok )
                glEnable( GL_POLYGON_SMOOTH );

        glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
        glDisable( GL_BLEND );

        glDepthMask( 1 );
        glDepthFunc( GL_LESS );
        
        glDisable( GL_DEPTH_TEST );
        glDisable( GL_ALPHA_TEST );

        glCullFace( GL_BACK );
        glEnable( GL_CULL_FACE );
        
        glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 2 " << (long) gl_error << endl;
        
        glEnable( GL_DEPTH_TEST );

        glShadeModel( GL_FLAT );

        //
        // draw the zone
        //
        //
        
        glColor3f( 0, 0.2, 0 );

        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 3 " << (long) gl_error << endl;

        drawZone( player );

        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 4 " << (long) gl_error << endl;
        
        glEnable( GL_CULL_FACE );
        
        const Vector<class Entity *> & vec = player->getContainerZone()->getContents();

        Vector< class Entity * >::const_iterator end = vec.end();

        for ( Vector< class Entity * >::const_iterator iter = vec.begin();
              iter != end; ++iter ) { 
                
                //
                // don't draw yourself!
                //
                
                if ( *iter == player )
                        continue;
                
                if ( (*iter)->die )
                        continue;

                class Entity * ent = *iter;
                
                glColor3f( 0.8, 0.8, 1 );
                glPushMatrix();
                {
                        glTranslatef( (*iter)->getPos()[0], (*iter)->getPos()[1], (*iter)->getPos()[2] );

                        const class Sprite * sprite = (*iter)->getSprite();

                        if ( sprite ) {
                                if ( ! sprite->clientSprite )
                                        sprite->clientSprite = new ClientSprite( sprite );
                                
                                showSprite( player, *iter );
                        }
                        
                        else {
                                glRotatef( ent->getAngle(), 0, 0, 1 );
                        
                                glPushMatrix();
                                
                                glRotatef( ent->getTilt()/8, 0, -1, 0 );
                        
                                const unsigned char * rgb = (*iter)->getIllumination();
                                glColor3ub( rgb[0], rgb[1], rgb[2] );
                                        
                                md2_model_t * my_model = ModelManager::getModel( ent->getModelName() );
                                class TextureObject * to  = TextureManager::getTexture( ent->getSkinName(), GL_RGBA );
                                        
                                class Character * ch = dynamic_cast< class Character *>( ent );

                                const float * trans = ent->getModelTrans();//my_model->getTrans( ent->getFrame() );
                                
                                glTranslatef( trans[0], trans[1], trans[2] );
                                        
                                //                                glTranslatef( 0, -trans[1], -trans[2] );

                                if ( my_model->isOk ) {
                                        drawModel( ent->getFrame(), *my_model, to, true );
                                                
                                        if ( ch ) {
                                                md2_model_t * my_weap_model = ModelManager::getModel( ch->getWeaponModelName() );
                                                class TextureObject * weap_to = TextureManager::getTexture( ch->getWeaponSkinName(), GL_RGBA );

                                                drawModel( ent->getFrame(), *my_weap_model, weap_to, true );
                                        }
                                }
                                        
                                glPopMatrix();
                                        
                                if ( draw_bounding || my_model->isOk == false ) {
                                        glDisable( GL_CULL_FACE );
                                        drawBounding( *iter, false, float(rgb[0])/255, float(rgb[1])/255, float(rgb[2])/255 );
                                        glEnable( GL_CULL_FACE );
                                }
                                
                        }
                        
                        ++g_state.num_ents;
                }
                glPopMatrix();


        }

        glPushMatrix();
        {

                const unsigned char * rgb = player->getIllumination();
                glColor4ub( rgb[0], rgb[1], rgb[2], 175 );
                
                if ( _cameraMode == FIRST_PERSON )  {
                        if ( draw_weapon ) {
                                glTranslatef( player->getPos()[0], 
                                              player->getPos()[1], 
                                              player->getPos()[2] + 24 );

                                glRotatef( player->getAngle(), 0, 0, 1 );
                                glTranslatef( 0, 0, 16 );
                                glRotatef( player->getTilt(), 0, -1, 0 );
                        
                                if ( player->getFrame() <= Character::FRAME_stand40 )
                                        glTranslatef( -8, 4, -8 );
                                else if ( player->getFrame() <= Character::FRAME_run6 )
                                        glTranslatef( -10, -4, -8 );
                                else if ( player->getFrame() <= Character::FRAME_attack8 )
                                        glTranslatef( -10, 4, -8 );
                        
                                glDisable( GL_DEPTH_TEST );
                                drawModel( player->getFrame(), *weapon_models[ cur_weapon ], weapon_textures[ cur_weapon ], true );
                                glEnable( GL_DEPTH_TEST );
                        }
                }
                else {
                        
                        md2_model_t * my_model = ModelManager::getModel( player->getModelName() );
                        md2_model_t * my_weap_model = ModelManager::getModel( player->getWeaponModelName() );
                        
                        class TextureObject * to  = TextureManager::getTexture( player->getSkinName(), GL_RGBA );
                        class TextureObject * weap_to  = TextureManager::getTexture( player->getWeaponSkinName(), GL_RGBA );

                        glTranslatef( player->getPos()[0],
                                      player->getPos()[1],
                                      player->getPos()[2] );

                        glRotatef( player->getAngle(), 0, 0, 1 );

                        glPushMatrix();
                        
                        glRotatef( player->getTilt()/8, 0, -1, 0 );
                        
                        const float * trans = my_model->getTrans( player->getFrame() );
                        
                        glTranslatef( trans[1]/4, 0, -trans[2] );

                        my_weap_model = weapon_models[ cur_weapon ];
                        weap_to  = weapon_textures[ cur_weapon ];
                        
                        drawModel( player->getFrame(), *my_weap_model, weap_to, true );
                        
                        drawModel( player->getFrame(), *my_model, to, true );

                        glPopMatrix();

                        if ( draw_bounding || my_model->isOk == false ) {
                                glDisable( GL_CULL_FACE );
                                drawBounding( player, false, float(rgb[0])/255, float(rgb[1])/255, float(rgb[2])/255 );
                                glEnable( GL_CULL_FACE );
                        }

                }
                //                        glTranslatef( 0, 0, -24 );
                //                        glCallList( _playerList );
        }
        glPopMatrix();


        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 5 " << (long) gl_error << endl;
        
        //
        // put the projection matrix back like we found it (ortho)
        //

        glMatrixMode( GL_PROJECTION );
        glPopMatrix();

        glDepthMask( 0 );
        glDisable( GL_DEPTH_TEST );

        glMatrixMode( GL_MODELVIEW );
                
        if ( draw_map_overlay ) {

                glLoadIdentity();
                
                glEnable( GL_BLEND );

                if ( g_state.polygon_smooth_ok )
                        glEnable( GL_POLYGON_SMOOTH );

                glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
                
                glColor4f( 0.0, 0.0, 0.0, 0.6 );
                glBegin( GL_QUADS );
                {
                        glVertex2f(            16, 16 );
                        glVertex2f( _size[0] - 16, 16 );
                        glVertex2f( _size[0] - 16, _size[1] - 16 );
                        glVertex2f(            16, _size[1] - 16 );
                }
                glEnd();

                glTranslatef( _size[0]/2, _size[1]/2, 0 );
                float scalefact = float( map_zoom_factor ) / MAP_ZOOM_BASE;
                
                glScalef( scalefact, scalefact, 0 );
                glPushMatrix();
                glRotatef( 90 - player->getAngle(), 0, 0, 1 );
                glTranslatef( -player->getPos()[0], -player->getPos()[1], 0 );

                drawZoneMap( player );
                
                glDisable( GL_BLEND );

                for ( Vector< class Entity * >::const_iterator iter = vec.begin();
                      iter != end; ++iter ) { 
                        
                        //
                        // don't draw yourself!
                        //
                        
                        if ( *iter == player )
                                continue;
                        
                        glColor3f( 0.8, 0.8, 1 );
                        glPushMatrix();
                        {
                                glTranslatef( (*iter)->getPos()[0], (*iter)->getPos()[1], (*iter)->getPos()[2] );
                                if ( *iter == player->getTargetEntity() )
                                        drawDisc( (*iter)->getBoundingRadius(), 1.0, 1.0, 0.8 );
                                else
                                        drawDisc( (*iter)->getBoundingRadius(), 0.6, 0.6, 0.8 );
                        }
                        glPopMatrix();
                }

                glPopMatrix();
                glCallList( automap_playerdisc_list );

                
        }
        
        glDisable( GL_FOG );
        glMatrixMode( GL_MODELVIEW );

        //
        // restore the viewport if needed
        //
        
        if ( viewport_scale != 10 )
                glViewport( viewport[0], viewport[1], viewport[2], viewport[3] );
        
        
        //        if ( ( gl_error = glGetError() ) != GL_NO_ERROR ) cout << " gl error 6 " << (long) gl_error << endl;


}

static void playSounds( const class Entity * ent, 
                        const class Character * listener ) {

        const float * lpos = listener->getPos();
        const float * pos = ent->getPos();
        float dist = -1.0;

        class Sound * sound = 0;

        Mix_Chunk * chunk = 0;
        
        const Vector< class Sound * > & sounds = ent->getSounds();
        for ( Vector< class Sound * >::const_iterator iter = sounds.begin();
              iter != sounds.end(); ++iter ) {

                sound = *iter;
                const class ClientSound * cs;
                
                if ( sound->die ) {
                        cs = ClientSound::findSound( sound->idnum );
                        if ( cs && ! Mix_Playing( cs->channel ) ) {
                                cs->terminate();
                        }
                        continue;
                }
                
                if ( dist < 0 ) {
                        dist = ( ent == listener ) ? 0.0F : VMath::distance3d( lpos, pos );
                }

                cs = ClientSound::addSound( sound->idnum, sound->volume, sound->loops );
                
                // null cs means we've run out of sound playing slots
                if ( cs == 0 )
                        return;
                
                switch ( sound->index ) {
                case Sound::SND_FIREBALL:
                        chunk = fire_wav;
                        break;
                case Sound::SND_EXPLOSION:
                        chunk = explode_wav;
                        break;
                case Sound::SND_LANDING:
                        chunk = turn_wav;
                        break;
                case Sound::SND_FOOTSTEP_SMALL:
                case Sound::SND_FOOTSTEP_MEDIUM:
                case Sound::SND_FOOTSTEP_BIG:
                        chunk = footstep_wavs[ footstep_map[ cur_footstep ] ];
                        
                        if ( ++cur_footstep >= num_mapped_footsteps ) {
                                cur_footstep = 0;
                        }
                        break;
                case Sound::SND_PAIN_FALL:
                        chunk = pain_fall_wav;
                        break;
                case Sound::SND_PAIN_MEDIUM:
                        chunk = pain2_wav;
                        break;
                case Sound::SND_SWORD_SLASH:
                        chunk = slash_wav;
                        break;
                case Sound::SND_HUMAN_MALE_JUMP:
                        chunk = jump_wav;
                        break;
                }
                
                if ( chunk == 0 ) {
                        cout << " :: Unhandled sound type " << (int) sound->index << endl;
                        sound->die = true;
                        cs->terminate();
                        continue;
                }
                
                if ( cs->channel < 0 ) {
                        cs->channel = Mix_PlayChannel( -1, chunk, cs->loops );
                        //                        cout << " starting idnum=" << cs->idnum << " on channel=" << cs->channel << endl;
                }
                
                if ( cs->channel < 0 ) {
                        continue;
                }
                
                if ( ! Mix_Playing( cs->channel ) ) {
                        cs->terminate();
                        continue;
                }
                
                static const int MAX_DIST = 3100;
                static const int MAX_DIST_SQ = MAX_DIST * MAX_DIST;

                int vol = int( VMath::square( MAX_DIST - dist ) * MIX_MAX_VOLUME * cs->volume / MAX_DIST_SQ / 255 );

                Mix_Volume( cs->channel, vol);
                cs->ok = true;
                
                if ( sound->loops >= 0 )
                        sound->die = true;
                
                break;
        }
}

static void updateSounds( const class Player * player, const float * pos ) {

        const Vector<class Entity *> vec = player->getContainerZone()->getContents();
        
        Vector< class Entity * >::const_iterator end = vec.end();

        ClientSound::tagSounds( false );
        
        for ( Vector< class Entity * >::const_iterator iter = vec.begin();
              iter != end; ++iter ) { 
                //                if ( *iter == player )
                //                        continue;
                playSounds( *iter, player );
        }
        
        ClientSound::purgeSounds( false );
}

void DungeonScreen::updateWidget() {

        class Player *player = Client::getPlayer();
        const float * pos = player->getPos();

        //
        //

        Uint32 tick_begin = SDL_GetTicks();

        player->getContainerZone()->physicsUpdateContents( SDL_GetTicks() );
        
        g_state.physics_time = SDL_GetTicks() - tick_begin;
        g_state.physics_fps = ( g_state.physics_time ? ( 1000 / g_state.physics_time ) : -1 );
        

        updateSounds( player, pos );

        
        Uint8 mouse_state = SDL_GetMouseState( 0, 0 );
        
        if ( pos[0] != last_pos[0] ||
             pos[1] != last_pos[1] ||
             pos[2] != last_pos[2] ) {
                
                //
                // first person view, bob head
                //
                /*
                  if ( _cameraMode == FIRST_PERSON  && 
                  ( pos[0] != last_pos[0] || pos[1] != last_pos[1] ) ) {

                  headbob += headbob_delta;
                  headbob_cur++;
                        
                  if ( headbob_cur >= headbob_max ) {
                  headbob_cur = -headbob_max;
                  headbob_delta = -headbob_delta;

                                //
                                // going back up, this is a footstep
                                //

                                if ( headbob_delta > 0 ) {
                                Mix_PlayChannel( -1, footstep_wavs[ footstep_map[ cur_footstep ] ], 0 );

                                if ( ++cur_footstep >= num_mapped_footsteps ) {
                                cur_footstep = 0;
                                }
                                }
                                }
                                }
                */
                //
                // third person view, follow the player with camera
                //

                last_pos[0] = pos[0];
                last_pos[1] = pos[1];
                last_pos[2] = pos[2];
        }
        
        if ( _cameraMode == THIRD_PERSON_FOLLOW ) {
                
                float angle_diff = VMath::angleDiff360( player->getAngle(), camera_angle_z );
                //                float angle_diff = player->getAngle() - camera_angle_z;
                
                if ( fabs( angle_diff ) >= 2 )
                        angle_diff /= 2;
                /*
                if ( angle_diff < -15 )
                        angle_diff = -15;
                else if ( angle_diff > 15 )
                        angle_diff = 15;
                */
                camera_angle_z = VMath::clampAngle360( camera_angle_z + angle_diff );



                angle_diff = VMath::angleDiff360( -( player->getTilt() + 180 ) / 2, camera_angle_x );
                
                if ( fabs( angle_diff ) >= 2 )
                        angle_diff /= 2;
                /*
                if ( angle_diff < -15 )
                        angle_diff = -15;
                else if ( angle_diff > 15 )
                        angle_diff = 15;
                */
                camera_angle_x = VMath::clampAngle360( camera_angle_x + angle_diff );

        }
                

        if ( ! getParent() ) {
                cout << " dungeon has no parent" << endl;
                return;
        }

        if ( ! player ) {
                cout << " dungeon has no player" << endl;
                return;
        }

        Uint8 * keystate = SDL_GetKeyState( 0 );
        SDLMod modstate = SDL_GetModState();
        
        if ( mouse_state & SDL_BUTTON( 1 ) ) {
                Client::getPlayer()->attack( 0 );
        }

        float self_prop[3] = { 0, 0, 0 };
        if ( mouse_state & SDL_BUTTON( 3 ) ) {
                self_prop[0] += 1;
        }
        
        if ( keystate[ SDLK_UP ] ) {
                if ( modstate & KMOD_CTRL ) {
                        if ( _cameraMode == FIRST_PERSON )
                                player->rotate( 0, 5 );
                        else
                                camera_angle_x += 5;
                }
                else {
                        self_prop[0] += 1;
                }
        }
        if ( keystate[ SDLK_DOWN ] ) {
                if ( modstate & KMOD_CTRL ) {
                        if ( _cameraMode == FIRST_PERSON )
                                player->rotate( 0, -5 );
                        else
                                camera_angle_x -= 5;
                }
                else {
                        self_prop[0] -= 1;
                }
        }

        if ( keystate[ SDLK_LEFT ] ) {
                if ( modstate & KMOD_CTRL ) {
                        camera_angle_z = VMath::clampAngle360( camera_angle_z - 5 );
                }
                else {
                        player->rotate( 5, 0 );
                }
        }
        if ( keystate[ SDLK_RIGHT ] ) {
                if ( modstate & KMOD_CTRL ) {
                        camera_angle_z = VMath::clampAngle360( camera_angle_z + 5 );
                }
                else {
                        player->rotate( -5, 0 );
                }
        }

        if ( keystate[ SDLK_a ] ) {
                self_prop[0] -= 1;
        }
        if ( keystate[ SDLK_s ] ) {
                self_prop[1] += 1;
        }
        if ( keystate[ SDLK_f ] ) {
                self_prop[1] -= 1;
        }

        if ( self_prop[0] || self_prop[1] ) {
                VMath::unitVector2d( self_prop );
        }
        
        int real_speed = speed;

        if ( modstate & KMOD_SHIFT ) {
                real_speed *= 3;
                real_speed /= 2;
        }

        player->setRelativeSelfPropulsion( self_prop[0] * real_speed , self_prop[1] * real_speed, 0 );

        if ( ! ( modstate & KMOD_CTRL )  ) {
                if ( keystate[ SDLK_RIGHTBRACKET ] ) zoom = max( 20, zoom - 10 );
                if ( keystate[ SDLK_LEFTBRACKET ] ) zoom = min( 600, zoom + 10 );
                
                if ( keystate[ SDLK_w ] && fov < 140 ) fov += 5;
                if ( keystate[ SDLK_q ] && fov > 5 ) fov -= 5;
        }


        playSounds( player, player );
}

// class MouseButtonListener
bool DungeonScreen::mousePressed(  SDL_MouseButtonEvent * evt ) {
        return false;
}
bool DungeonScreen::mouseReleased( SDL_MouseButtonEvent * evt ) {
        return false;
}
// class MouseMotionListener {
bool DungeonScreen::mouseMoved(    SDL_MouseMotionEvent * evt ) {

        static int center[2] = { 
                _size[0] / 2, 
                _size[1] / 2 
        };
        
        class Player *player = Client::getPlayer();
        
        if (  ( evt->x != center[0] || evt->y != center[1] )  ) {
                float rot[2] = {
                        float( evt->xrel ) * fov / 90,
                        float( evt->yrel ) * fov / 90
                };

                if ( _cameraMode != FIRST_PERSON ) {
                        rot[0] /= 2;
                        rot[1] /= 2;
                }
                        
                player->rotate( - rot[0], rot[1] );
                //                camera_angle_x = min( max( camera_angle_x - rot[1], float( -180.0 ) ), float( 0.0 ) );

                        //                camera_angle_x = VMath::clampAngle360( camera_angle_x - rot[1] );
                SDL_WarpMouse( center[0], center[1] );
                
                /*
                player->getContainerZone()->setTargetEntity( player, 
                                                             VMath::degrees2radians( player->getAngle() ),
                                                             VMath::degrees2radians( player->getTilt()  ) );
                */
                
        }

        playSounds( player, player );

        return false;
}
bool DungeonScreen::mouseEntered(  SDL_MouseMotionEvent * evt ) {
        return false;
}
bool DungeonScreen::mouseExited(   SDL_MouseMotionEvent * evt ) {
        return false;
}

bool DungeonScreen::keyReleased( SDL_KeyboardEvent * evt ) {
        if ( GenericScreen::keyReleased( evt ) )
                return true;
        
        return false;
        
}

bool DungeonScreen::keyPressed(   SDL_KeyboardEvent * evt, int x, int y ) {
        if ( GenericScreen::keyPressed( evt, x, y ) )
                return true;
        
        class Player *player = Client::getPlayer();
        SDLMod modstate = SDL_GetModState();

        switch ( evt->keysym.sym ) {
        case SDLK_0:
                cur_weapon = 0;
                break;
        case SDLK_1:
                cur_weapon = 1;
                break;
        case SDLK_2:
                cur_weapon = 2;
                break;
        case SDLK_3:
                cur_weapon = 3;
                break;
        case SDLK_4:
                cur_weapon = 4;
                break;
        case SDLK_5:
                cur_weapon = 5;
                break;
        case SDLK_6:
                cur_weapon = 6;
                break;
        case SDLK_7:
                cur_weapon = 7;
                break;
        case SDLK_8:
                cur_weapon = 8;
                break;
        case SDLK_9:
                cur_weapon = 9;
                break;
        case SDLK_p:
                speed += 2;
                break;
        case SDLK_o:
                speed -= 2;
                break;

                //
                // toggle camera
                //

        case SDLK_F9:
                switch ( _cameraMode ) {
                case FIRST_PERSON:
                        _cameraMode = THIRD_PERSON_FOLLOW;
                        break;
                case THIRD_PERSON_FOLLOW:
                        _cameraMode = THIRD_PERSON;
                        camera_angle_z = player->getAngle();
                        camera_angle_x = -60;
                        break;
                case THIRD_PERSON:
                        _cameraMode = FIRST_PERSON;
                        //                        camera_angle_x = -90;
                        g_state.show_cursor = false;
                        break;
                }
                
                break;

        case SDLK_TAB:
                draw_map_overlay = ! draw_map_overlay;
                break;

        case SDLK_EQUALS:
                if ( draw_map_overlay ) {
                        if ( map_zoom_factor < MAP_ZOOM_BASE )
                                map_zoom_factor <<= 1;
                }
                else {
                        if ( viewport_scale < 10 )
                                ++viewport_scale;
                }

                break;

        case SDLK_MINUS:
                if ( draw_map_overlay ) {
                        if (  map_zoom_factor > 1 )
                                map_zoom_factor >>= 1;
                }
                else {
                        if ( viewport_scale > 1 )
                                --viewport_scale;
                }
                break;
                        
        case SDLK_f:
                if ( modstate & KMOD_CTRL ) {
                        do_fill = ! do_fill;
                }
                break;

        case SDLK_b:
                if ( modstate & KMOD_CTRL ) {
                        draw_bounding = ! draw_bounding;
                }
                break;
        case SDLK_w:
                if ( modstate & KMOD_CTRL ) {
                        draw_weapon = ! draw_weapon;
                }
                break;

        case SDLK_y:
                if ( modstate & KMOD_CTRL ) {
                        player->toggleFlag( Entity::FLYING );
                }
                break;

        case SDLK_d:
                if ( modstate & KMOD_CTRL ) {
                        class Zone * zone = player->getContainerZone();
                        
                        zone->setUseDynamicLights( ! zone->getUseDynamicLights() );
                }
                break;

        case SDLK_t:
                if ( modstate & KMOD_CTRL ) {
                        draw_textures = !draw_textures;
                }
                break;

        case SDLK_l:
                if ( modstate & KMOD_CTRL ) {
                        draw_lightmaps = !draw_lightmaps;
                }
                break;

        case SDLK_v:
                if ( modstate & KMOD_CTRL ) {
                        draw_using_pvs = !draw_using_pvs;
                }
                break;

        case SDLK_i:
                if ( modstate & KMOD_CTRL ) {
                        draw_invisible_sectors = !draw_invisible_sectors;
                }
                break;

        case SDLK_UP:

                if ( modstate )
                        break;

                //                player->setRelativeSelfPropulsion( 10, 0, 0 );
                break;

        case SDLK_DOWN:

                if ( modstate )
                        break;

                //                player->setRelativeSelfPropulsion( -10, 0, 0 );
                break;

        case SDLK_SPACE:
        case SDLK_BACKSPACE:
                player->jump();
                //                player->addRelativeSelfPropulsion( 0, 0, 48 );
                break;
        default:
                break;
        }

        playSounds( player, player );

        return false;
}

