/*
 * server/Server.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "Server.h"
#include <iostream>
#include <cstdlib>

#include "Peer.h"
#include "Player.h"


/**
 * this constructor assumes that SDL and SDL_net have been initialized
 */

Server::Server( int port )
        : _thread( 0 )
{

        //
        // Allocate the socket set
        //
	_socketset = SDLNet_AllocSocketSet( MAX_CLIENTS );
        
	if ( _socketset == 0 ) {
                cout << "Couldn't create socket set: " << SDLNet_GetError() << endl;
                return;
	}
		
	//
        // Create the server socket
        //
        
	SDLNet_ResolveHost( & _serverIP, 0, port );
        
	_servsock = SDLNet_TCP_Open( & _serverIP );
        
	if ( _servsock == NULL ) {
		cout << "Couldn't create server socket: " << SDLNet_GetError() << endl;
                return;
	}

	SDLNet_TCP_AddSocket( _socketset, _servsock );  
        
}

Server::~Server() {
        
}

static int threadrun( void * ptr ) {
        ( ( Server * ) ptr )->run( false );
        cout << " :: thread ending" << endl;
        return 0;
}

bool Server::handleNewConnection() {

        TCPsocket newsock;
        
	newsock = SDLNet_TCP_Accept( _servsock );
        
	if ( newsock == NULL ) {
                cout << " :: Server: failed to create new socket on peer connection." << endl;
		return false;
	}

        cout << " :: Server: new connection" << endl;

        if ( _peers.size() >= MAX_CLIENTS ) {
                cout << "MAX_CLIENTS connected, new connection refused." << endl;

                SDLNet_TCP_Send( newsock, 
                                 "MAX_CLIENTS reached, Connection refused.\n",
                                 41 );
                
                SDLNet_TCP_DelSocket( _socketset,
                                      newsock );
                SDLNet_TCP_Close( newsock );
        }

        class Peer * p = new Peer();
        p->active = true;
        p->sock = newsock;
        p->addr = *SDLNet_TCP_GetPeerAddress( newsock );
        p->_player = new Player( "A Player", "dummy" );
        p->_player->setPeer( p );

        strcpy( p->name, "a client" );
        SDLNet_TCP_AddSocket( _socketset, newsock );

        _peers.push_back( p );

        return true;
}

bool Server::handlePeer( class Peer * peer ) {
        char data[512];

        int num_read = SDLNet_TCP_Recv( peer->sock, data, 511 );

        //
	// Has the connection been closed?
        //

	if ( num_read <= 0 ) {
                cout <<  " :: Server: Closing peer socket: " << peer->active << endl;
                
		//
                // todo: Notify all active clients */
                //

                peer->active = false;
		SDLNet_TCP_DelSocket( _socketset, peer->sock );
		SDLNet_TCP_Close( peer->sock );
                peer->sock = 0;
                return false;

	} 
        else {
                data[ num_read ] = 0;
                
                cout << " got: " << data << endl;

                if ( peer->_player ) {
                        peer->_player->handleInput( data, num_read );
                }
        }

        return true;
}

void Server::run( bool spawnthread ) {

        if ( spawnthread && _thread == 0 ) {
                _thread = SDL_CreateThread( threadrun, this );
                return;
        }

        if ( _socketset == 0 || _servsock == 0 ) {
                cout << " :: Server: Server cannot run with incomplete initialization." << endl;
                return;
        }

        cout << " :: running server" << endl;
        
        class Vector<class Peer *> purgePeers;
        
        while ( true ) {
                SDL_Delay( 10 );

                //
                // Wait for events 
                //

		SDLNet_CheckSockets( _socketset, ~0 );

		//
                // Check for new connections
                //

		if ( SDLNet_SocketReady( _servsock ) ) {
                        handleNewConnection();
		}
                
                //
                // read incoming data 
                //
                
                purgePeers.clear();
                
                Vector< class Peer *>::const_iterator end = _peers.end();
                for( Vector< class Peer *>::const_iterator iter = _peers.begin();
                     iter != end; ++iter ) {                
                        cout << " testing " << *iter << endl;

                        if ( SDLNet_SocketReady( (*iter)->sock ) ) {
                                handlePeer( *iter );
                        }

                        //
                        // mark peer for purge
                        //

                        if ( ! (*iter)->active ) {
                                purgePeers.push_back( *iter );
                        }
                }
                
                //
                // purge all inactive  connections
                //

                end = purgePeers.end();
                for( Vector< class Peer *>::const_iterator iter = purgePeers.begin();
                     iter != end; ++iter ) {
                        cout << " :: Server: purging " << *iter << endl;
                        _peers.remove( *iter );
                }
        }
}

