/*
 * server/Zone_light.cc: methods for static lighting
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */


#include <cstdio>
#include <cmath>
#include <string>
#include <iostream>
#include <iomanip>
#include <fstream>

#include "Zone.h"
#include "../util/VMath.h"
#include "Sector.h"
#include "SectorSpecial.h"
#include "LineDef.h"
#include "SideDef.h"
#include "Lightmap.h"
#include "LightSource.h"

static Array< Array<LightSource *> * > sectorLights;

static void dumpLightmap( const char *filename, int num, const class Lightmap * lm ) {
        char fname[1024];
        sprintf( fname, "%s_%d.ppm", filename, num );
        
        ofstream ofs( fname );
        
        ofs << "P6" << endl
            << (int) lm->size[0] << " " << (int) lm->size[1] << endl
            << "255" << endl;
        ofs.write( lm->static_pixels, (lm->size[0] * lm->size[1] * 3) );
        ofs.close();
}

void Zone::resetLighting( const class Entity * ent ) {
}

bool Zone::illuminateSectors( unsigned int sample_size, 
                              unsigned int texels, 
                              float scatter_threshold,
                              float scatter_minval,
                              vector<class Lightmap *> & blm ) {

        cout << " :: illuminating sector floor/ceiling pairs" << endl;

        class Lightmap * lms[2] = { 0, 0 };
        
        for ( int i = 0; i < _sectors.length; ++i ) {
                
                class Sector * s = _sectors[i];
                
                //
                // create the lightmaps if needed
                //

                if ( ! s->floor_lightmap ) {
                        s->floor_lightmap = new Lightmap();
                        blm.push_back( s->floor_lightmap );
                }
                
                if ( ! s->ceiling_lightmap ) {
                        s->ceiling_lightmap = new Lightmap();
                        blm.push_back( s->ceiling_lightmap );
                }
                
                lms[0] = s->floor_lightmap;
                lms[1] = s->ceiling_lightmap;

                lms[0]->setSize( s->_extent[0], s->_extent[1] );
                lms[1]->setSize( s->_extent[0], s->_extent[1] );

                for ( int j = 0; j < 2; ++j ) {
                        if ( lms[j]->static_pixels ) {
                                delete[] lms[j]->static_pixels;
                        }
                        lms[j]->static_pixels = new unsigned char[ lms[j]->size[0] * lms[j]->size[1] * 3 ];
                        
                        lms[j]->clear( 0, 0, 0 );
                }
                
                //
                // convert the extents to be in units of sample_size
                //

                int sample_extents[2] = {
                        (int) ( s->_extent[0] ) / Lightmap::TEX_RES * sample_size,
                        (int) ( s->_extent[1] ) / Lightmap::TEX_RES * sample_size
                };

                //
                // dd is half of the actual sample size, used to find the
                // actual surface point to sample against
                //
                
                float d = float( Lightmap::TEX_RES ) / sample_size;
                float dd = d / 2;
                
                //
                // the fraction of light delivered to each sample surface point
                // which is applied to the corresponding lightmap texel
                // e.g. if the sample size is half the texel size in both x and y, t
                // he fraction of light will be 1/4
                //
                
                int   sample_divisor  = ( sample_size * sample_size );
                
                class Array<LightSource *> & sLights = *sectorLights[ i ];
                
                cout << " :: illuminating sector " << setw(3) << s->_num << " : " 
                     << setw(3) << (int)lms[0]->size[0] << "x" << setw(3) << (int) lms[0]->size[1] 
                     << " lightmap(s): [   %]\b\b" << flush;

                //
                // iteratate along the T-direction
                //

                for ( int y = 0; y < sample_extents[1]; ++y ) {
                        
                        int pct = y * 100 / sample_extents[1];
                        cout << "\b\b\b" << setw(3) << pct << flush;
                        
                        float surfPt[3] = {
                                0,
                                s->_offset[1] + d * y + dd,
                                0,
                        };
                        
                        int lightmap_pixel = ( y / sample_size ) * lms[0]->size[0];

                        for ( int x = 0; x < sample_extents[0]; ++x ) {
                                
                                int my_lightmap_pixel = ( lightmap_pixel + ( x / sample_size ) ) * 3;
                                
                                unsigned char * p[2] = {
                                        lms[0]->static_pixels + my_lightmap_pixel,
                                        lms[1]->static_pixels + my_lightmap_pixel
                                };

                                for ( int light_i = 0; light_i < sLights.length; ++light_i ) {

                                        const class LightSource * lt = sLights[ light_i ];
                                        
                                        const float * lightPt = lt->getPos();
                                        
                                        //
                                        // do the floor first
                                        //

                                        surfPt[0] = s->_offset[0] + d * x + dd;
                                        surfPt[2] = s->getFloorHeight();
                                        
                                        if ( fabs( surfPt[0] - lightPt[0] ) > lt->getIntensity() ||
                                             fabs( surfPt[1] - lightPt[1] ) > lt->getIntensity() )
                                                continue;

                                        float vals[2] = { 0, 0 };
                                        float blocked_dist;
                                        
                                        //
                                        // lightpoints that lie below the floor do not illuminate this surface
                                        //

                                        if ( surfPt[2] > lightPt[2] || isRayBlocked( surfPt,
                                                                                     lightPt,
                                                                                     lt->_sect,
                                                                                     0,
                                                                                     s,
                                                                                     & blocked_dist ) ) {
                                                vals[0] = 0;
                                        }
                                
                                        else {
                                                float dist = VMath::distance3d( surfPt, lightPt );
                                                vals[0] = max( float(0), lt->getIntensity() - dist );

                                                if ( scatter_threshold > 0 && blocked_dist >= 0 && 
                                                     blocked_dist <= scatter_threshold )
                                                        vals[0] -= 
                                                                vals[0] *
                                                                ( scatter_threshold - blocked_dist ) * 
                                                                scatter_minval / 
                                                                scatter_threshold;
                                        }

                                        //
                                        // do the ceiling
                                        // 

                                        surfPt[2] = s->getCeilingHeight();

                                        //
                                        // lightpoints that lie above the ceiling do not illuminate this surface
                                        //

                                        if ( surfPt[2] < lightPt[2] || isRayBlocked( surfPt,
                                                                                     lightPt,
                                                                                     lt->_sect,
                                                                                     0,
                                                                                     s,
                                                                                     & blocked_dist ) ) {
                                                vals[1] = 0;
                                        }
                                
                                        else {
                                                float dist = VMath::distance3d( surfPt, lightPt );
                                                vals[1] = max( float(0), lt->getIntensity() - dist );

                                                if ( scatter_threshold > 0 && blocked_dist >= 0 && 
                                                     blocked_dist <= scatter_threshold )
                                                        vals[1] -= 
                                                                vals[1] * 
                                                                ( scatter_threshold - blocked_dist ) * 
                                                                scatter_minval / 
                                                                scatter_threshold;
                                        }
                                
                                        if ( vals[0] <= 0 && vals[1] <= 0 )
                                                continue;
                                        
                                        for ( int i = 0; i < 3; ++i ) {
                                                for ( int j = 0; j < 2; ++j ) {
                                                        if ( p[j][i] < lt->getRGB()[i] ) {
                                                                float addit = ( vals[j] * lt->getRGB()[i] ) / 255 / sample_divisor;
                                                                unsigned char newval = min( p[j][i] + (int)addit, 255 );
                                                                newval = min( newval, lt->getRGB()[i] );
                                                                p[j][i] = (unsigned char) newval;
                                                        }
                                                }
                                        }
                                }
                        }
                }

                /**
                 * code to dump out the lightmap
                 */
                
                if ( false ) {
                        dumpLightmap( "gen_lm_floor", s->_num, lms[0] );
                        dumpLightmap( "gen_lm_ceil", s->_num, lms[1] );
                }                
                
                cout << "\b\b\b" << 100 << "%] -- Done." << endl;

        }

        return true;
}

void Zone::illuminateWall( class Lightmap * lm, 
                           const class LineDef * line,
                           const class LightSource * lt,
                           const float * origin, 
                           unsigned int sample_size,
                           const float * extents ) {

        //
        // half-sizes in world space
        //
        
        int sample_extents[2] = {
                int ( line->length ) / Lightmap::TEX_RES * sample_size,
                int ( extents[2] ) / Lightmap::TEX_RES * sample_size,
        };

        float dx = extents[0] / sample_extents[0];
        float dy = extents[1] / sample_extents[0];
        float dz = extents[2] / sample_extents[1];

        float ddx = dx / 2;
        float ddy = dy / 2;
        float ddz = dz / 2;
        
        float sample_divisor = VMath::square( sample_size );

        const float * lightPt = lt->getPos();
        
        for ( int t = 0; t < sample_extents[1]; ++t ) {
                
                int lightmap_pixel = ( t / sample_size ) * lm->size[0];
                        
                for ( int s = 0; s < sample_extents[0]; ++s ) {
                        
                        float surfPt[3] = {
                                origin[0] + dx * s + ddx,
                                origin[1] + dy * s + ddy,
                                origin[2] + dz * t + ddz
                        };
                        
                        if ( fabs( surfPt[0] - lightPt[0] ) > lt->getIntensity() ||
                             fabs( surfPt[1] - lightPt[1] ) > lt->getIntensity() ||
                             fabs( surfPt[2] - lightPt[2] ) > lt->getIntensity() )
                                continue;
                        
                        float val = 0;

                        float blocked_dist;
                        
                        if ( isRayBlocked( surfPt,
                                           lightPt,
                                           lt->_sect,
                                           line,
                                           0,
                                           & blocked_dist ) ) {
                                val = 0;
                        }
                                
                        else {
                                float dist = VMath::distance3d( surfPt, lightPt );
                                val = max( float(0), lt->getIntensity() - dist );
                        }
                                
                        if ( val <= 0 )
                                continue;

                        int my_lightmap_pixel = ( lightmap_pixel + ( s / sample_size ) ) * 3;
                        
                        unsigned char * p = lm->static_pixels + my_lightmap_pixel;
                        
                        for ( int rgb_i = 0; rgb_i < 3; ++rgb_i ) {
                                if ( *p < lt->getRGB()[ rgb_i ] ) {
                                        float addit = ( val * lt->getRGB()[ rgb_i ] ) / sample_divisor / 255;
                                        unsigned char newval = min( *p + (int) addit, 255 );
                                        newval = min( newval, lt->getRGB()[ rgb_i ] );
                                        *p = (unsigned char) newval;
                                }
                                ++p;
                        }
                }
        }
}

inline static void illuminateWallSections( class Zone * zone,
                                    class SideDef * side,
                                    class SideDef * other,
                                    const class LightSource * lt,
                                    const int sample_size,
                                    const float  * origin,
                                    float * extents
                                    ) {
        
        static int lnum = 0;

        float origin3d[3] = { 
                origin[0],
                origin[1],
                0
        };

        for ( int section_i = 0; section_i < 3; ++section_i ) {
                class Lightmap * lm = side->lightmaps[ section_i ];
                
                if ( ! lm )
                        continue;

                extents[2] = side->getWallSectionHeight( other, section_i );
                origin3d[2]  = side->getWallSectionZOrigin( other, section_i );
  
                if ( other && other->sector->spec ) {
                        if ( section_i == SideDef::UPPER &&
                             other->sector->spec->isDoor() ) {
                                extents[2]  += other->sector->spec->travel;
                                origin3d[2] -= other->sector->spec->travel;
                        }
                        else if ( section_i == SideDef::LOWER && other->sector->spec->isLift() ) {
                                extents[2] += other->sector->spec->travel;
                        }
                }

                zone->illuminateWall( lm, side->line, lt, origin3d, sample_size, extents );
                
                if ( false ) {
                        dumpLightmap( "gen_lm_side_front", lnum++, lm );
                }
        }
}

static void allocateWallLightmap( class SideDef * side, 
                                 class SideDef * other, 
                                 int section_i,
                                 vector<class Lightmap *> & blm  ) {

        class Lightmap * lm = 0;

        if ( ! side->lightmaps[ section_i ] ) {
                lm = new Lightmap();

                int height = side->getWallSectionHeight( other, section_i );
                
                if ( other && other->sector->spec &&
                     ( ( section_i == SideDef::UPPER &&
                         other->sector->spec->isDoor() ) ||
                       ( section_i == SideDef::LOWER &&
                         other->sector->spec->isLift() ) ) ) {
                        height += other->sector->spec->travel;
                }

                lm->setSize( side->line->length, height );
                blm.push_back( lm );
                                        
                side->lightmaps[ section_i ] = lm;
        }
        else {
                lm = side->lightmaps[ section_i ];
        }

        if ( lm->static_pixels ) {
                delete[] lm->static_pixels;
        }
                                
        lm->static_pixels = new unsigned char[ lm->size[0] * lm->size[1] * 3 ];

        lm->clear( 0, 0, 0 );
}

bool Zone::illuminateWalls( unsigned int sample_size,
                            unsigned int texels, 
                            vector<class Lightmap *> & blm  ) {
        cout << " :: illuminating walls: [   %]\b\b" << flush;
        
        //        class Lightmap * lm = 0;

        for ( int line_i = 0; line_i < _lines.length; ++line_i ) {

                int pct = line_i * 100 / _lines.length;
                cout << "\b\b\b" << setw(3) << pct << flush;
                
                class LineDef * line = _lines[ line_i ];

                class SideDef * frontSide = line->sides[0];
                class SideDef * backSide = line->sides[1];

                //
                // first create or clear the lightmaps on the side
                //

                for ( int section_i = 0; section_i < 3; ++section_i ) {
                        
                        //                        cout << " testing segment " << i << endl;
                        
                        if ( frontSide->hasWallSection( backSide, section_i ) ) {
                                allocateWallLightmap( frontSide, backSide, section_i, blm );
                        }

                        if ( backSide && backSide->hasWallSection( frontSide, section_i ) ) {
                                allocateWallLightmap( backSide, frontSide, section_i, blm );
                        }
                }
                
                float extents[3] = {
                        line->verts[1][0] - line->verts[0][0],
                        line->verts[1][1] - line->verts[0][1],
                        0,
                };

                float neg_extents[3] = {
                        - line->verts[1][0] + line->verts[0][0],
                        - line->verts[1][1] + line->verts[0][1],
                        0,
                };

                for ( int light_i = 0; light_i < _lights.length; ++light_i ) {

                        const class LightSource * lt = _lights[ light_i ];
                        
                        const float * lightPt = lt->getPos();

                        //
                        // illuminate the front of the line
                        //

                        if ( VMath::whichSide( line->verts[0], line->verts[1], lightPt ) == 0 ) {

                                illuminateWallSections( this,
                                                        frontSide,
                                                        backSide,
                                                        lt,
                                                        sample_size,
                                                        line->verts[0],
                                                        extents );
                        }

                        //
                        // illuminate the back of the line
                        //
                        
                        else if ( backSide ) {
                
                                illuminateWallSections( this,
                                                        backSide,
                                                        frontSide,
                                                        lt,
                                                        sample_size,
                                                        line->verts[1],
                                                        neg_extents );
                               
                        }
                }
        }

        cout << "\b\b\b" << 100 << "%] -- Done." << endl;

        return true;
}

bool Zone::illuminate( unsigned int sample_size, 
                       unsigned int texels,
                       float scatter_threshold,
                       float scatter_minval ) {
        
        cout << " :: illuminating map with texels " << texels << endl;

        for ( int i = 0; i < _lights.length; ++i ) {
                _lights[i]->_sect = findSector( _lights[i]->getPos() );
        }
        
        sectorLights.setSize( _sectors.length );
        
        for ( int i = 0; i < _sectors.length; ++i ) {
                
                vector< LightSource *> myLights;
                
                for ( int j = 0; j < _lights.length; ++j ) {
                        if ( _lights[j]->_sect == _sectors[i] ||
                             ( _lights[j]->_sect && sectorsCanSee( _lights[j]->_sect, _sectors[i] ) ) )
                                myLights.push_back( _lights[j] );
                }
                
                sectorLights[i] = new Array< LightSource *>( myLights.size() );
                
                vector< LightSource *>::const_iterator end = myLights.end();
                
                int num = 0;
                for ( vector< LightSource *>::const_iterator iter = myLights.begin();
                      iter != end; ++iter ) {
                        (*sectorLights[i])[ num++ ] = *iter;
                }
        }

        vector<class Lightmap *> blm;
        
        illuminateSectors( sample_size, texels, 
                           scatter_threshold, scatter_minval, 
                           blm );

        illuminateWalls( sample_size, texels, blm );

        cout << " :: illumination complete" << endl;
        
        if ( blm.size() ) {

                int old_len = _lightmaps.length;

                cout << " :: expanding _lightmaps array from " << old_len << " by " << blm.size() << endl;
                
                _lightmaps.setSize( old_len + blm.size() );
                
                for ( unsigned int i = 0; i < blm.size(); ++i ) {
                        _lightmaps[ i + old_len ] = blm[i];
                }
        }

        for ( int i = 0; i < _lightmaps.length; ++i ) {
                _lightmaps[i]->clampEdges( _lightmaps[i]->static_pixels );
        }
        
        return true;
}
/**
 *
 * cast a ray from pt1 to pt2.
 * with illumination, the pt1 is typically the surface point and pt2 is typically the light point
 * (contrary to common sense)
 *
 * blocked_dist has 2 meanings:  if a ray is blocked, then blocked_dist is the distance from the
 * top or bottom of the blocking edge.  basically it is a measure of how deeply a ray cuts into an
 * edge.
 * if a ray is not blocked, it is a measure of how close the ray passed to the closest edge.
 * it can be used for a pseudo light-scatter effect
 * if blocked_dist is negative, then its value is not useful.  blocked_dist will be negative
 * for rays cast from sector surfaces to a light that is contained in the sector.  all other cases
 * should render non-negative values
 * 
 */
bool Zone::isRayBlocked( const float * pt1, const float * pt2,
                         const class Sector * target_sector,
                         const class LineDef * self_line,
                         const class Sector * self_sector,
                         float * blocked_dist ) const {

        float u[2];

        if ( pt1[0] == pt2[0] && pt1[1] == pt2[1] )
                return false;
        
        float slope = ( pt2[2] - pt1[2] );
        
        bool was_in_self_sector = ( self_sector ? ( self_sector == target_sector ) : false );
        
        *blocked_dist = -1;

        for ( int i = 0; i < _lines.length; ++i ) {
                const class LineDef * line = _lines[i];
                
                if ( line == self_line )
                        continue;
                
                if ( self_sector ) {
                        if ( line->sides[0]->sector == self_sector ) {
                                if ( VMath::whichSide( line->verts[0], line->verts[1], pt2 ) == 0 ) {
                                        continue;
                                }
                        }
                        
                        else if ( line->sides[1] && line->sides[1]->sector == self_sector ) {
                                if ( VMath::whichSide( line->verts[1], line->verts[0], pt2 ) == 0 ) {
                                        continue;
                                }
                        }
                }
                
                if ( VMath::lineSegmentsIntersect2d( pt1,
                                                     pt2,
                                                     line->verts[0],
                                                     line->verts[1],
                                                     &u[0], &u[1] ) ) {
                        
                        if ( line->sides[0]->sector == self_sector ||
                             ( line->sides[1] && line->sides[1]->sector == self_sector ) ) {
                                was_in_self_sector = true;
                        }
                        
                        if ( self_line && u[0] == 0 || u[0] == 1 ) {
                                // let back-facing walls block the ray at endpoints

                                continue;
                        }

                        if ( line->sides[1] ) {
                                
                                float crossZ = u[0] * slope + pt1[2];
                                
                                float fuzz[2];
                                
                                if ( crossZ < line->sides[0]->sector->getFloorHeight() ||
                                     crossZ < line->sides[1]->sector->getFloorHeight() ) {
                                        *blocked_dist = max( line->sides[0]->sector->getFloorHeight() - crossZ,
                                                             line->sides[1]->sector->getFloorHeight() - crossZ );
                                        return true;
                                }
                                else {
                                        fuzz[0] = min( crossZ - line->sides[0]->sector->getFloorHeight(),
                                                       crossZ - line->sides[1]->sector->getFloorHeight());
                                }
                                                    
                                if ( crossZ > line->sides[0]->sector->getCeilingHeight() ||
                                     crossZ > line->sides[1]->sector->getCeilingHeight() ) {
                                        *blocked_dist = max( crossZ - line->sides[0]->sector->getCeilingHeight(),
                                                             crossZ - line->sides[1]->sector->getCeilingHeight());
                                        return true;
                                }
                                else {
                                        fuzz[1] = min( line->sides[0]->sector->getCeilingHeight() - crossZ,
                                                       line->sides[1]->sector->getCeilingHeight() - crossZ );
                                }
                                
                                if ( *blocked_dist < 0 )
                                        *blocked_dist = 999;
                                
                                *blocked_dist = min( *blocked_dist, min( fuzz[0], fuzz[1] ) );
                                
                                continue;
                        }

                        return true;
                }
        }
        
        if ( self_sector && was_in_self_sector == false ) {
                const Sector * osect = findSector( pt1 );
                if ( osect && self_sector != osect ) {
                        if ( ( slope > 0 && osect->getFloorHeight() > self_sector->getFloorHeight() ) ||
                             ( slope < 0 && osect->getCeilingHeight() < self_sector->getCeilingHeight() ) ) {
                                return true;
                        }
                }
        }

        return false;
}

