/*
 * widgets/ScrollpaneWidget.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "ScrollpaneWidget.h"
#include "ScrollbarWidget.h"
#include "Viewport.h"

/**
 * default horizScrollPolicy = SCROLLBAR_AS_NEEDED,
 * vertScrollPolicy = SCROLLBAR_AS_NEEDED,
 *  horizScrollbarThickness = 16,
 *  vertScrollbarThickness = 16
 */

ScrollpaneWidget::ScrollpaneWidget( int w, int h,
                                    class Widget * scrollable,
                                    enum ScrollPolicy horizScrollPolicy,
                                    enum ScrollPolicy vertScrollPolicy,
                                    int horizScrollbarThickness,
                                    int vertScrollbarThickness ) : Widget( w, h ) {

        _name = "ScrollpaneWidget";

        setOpaque( true );

        _scrollPolicy[ 0 ] = horizScrollPolicy;
        _scrollPolicy[ 1 ] = vertScrollPolicy;

        bool scrollbarOk[2] = { false, false };

        if ( horizScrollbarThickness > 0 &&
             ( _scrollPolicy[ 0 ] == SCROLLBAR_ALWAYS ||
               ( _scrollPolicy[ 0 ] == SCROLLBAR_AS_NEEDED && scrollable->getWidth() > _size[ 0 ] ) ) ) {
                scrollbarOk[ 0 ] = true;
        }

        if ( vertScrollbarThickness > 0 &&
             ( _scrollPolicy[ 1 ] == SCROLLBAR_ALWAYS ||
               ( _scrollPolicy[ 1 ] == SCROLLBAR_AS_NEEDED && scrollable->getHeight() > _size[ 1 ] ) ) ) {
                scrollbarOk[ 1 ] = true;
        }

        _scrollViewport = new Viewport( _size[0] - ( scrollbarOk[ 1 ] ? vertScrollbarThickness : 0 ),
                                  _size[1] - ( scrollbarOk[ 0 ] ? horizScrollbarThickness : 0 ),
                                  scrollable );

        //        cout << " _scrollViewport: " << _scrollViewport << ", scrollable " << scrollable << ", view " << _scrollViewport->getView() << endl;

        if ( scrollbarOk[ 0 ] ) {
                _scrollbars[ 0 ] = new ScrollbarWidget( _scrollViewport->getWidth(),
                                                        horizScrollbarThickness, 
                                                        SCROLL_HORIZONTAL,
                                                        _scrollViewport );
                add( _scrollbars[ 0 ], 0, 0 );
        }
        else {
                _scrollbars[ 0 ] = 0;
        }

        if ( scrollbarOk[ 1 ] ) {
                _scrollbars[ 1 ] = new ScrollbarWidget( vertScrollbarThickness, 
                                                        _scrollViewport->getHeight(),
                                                        SCROLL_VERTICAL,
                                                        _scrollViewport );
                add( _scrollbars[1], _size[ 0 ] - vertScrollbarThickness, 
                     ( scrollbarOk[ 0 ] ? horizScrollbarThickness : 0 ) );
        }
        else {
                _scrollbars[ 1 ] = 0;
        }

        add( _scrollViewport, 0, ( scrollbarOk[ 0 ] ? horizScrollbarThickness : 0 ) );
}

/**
 * the Viewport is deleted, and any existing views may have dangling references to it
 */
ScrollpaneWidget::~ScrollpaneWidget() {
        if ( _scrollbars[0] )
                delete _scrollbars[0];
        if ( _scrollbars[1] )
                delete _scrollbars[1];

        delete _scrollViewport;
}

bool ScrollpaneWidget::keyPressed( SDL_KeyboardEvent * evt, int x, int y ) { 
        
        if ( evt->keysym.sym == SDLK_PAGEUP || evt->keysym.sym == SDLK_PAGEDOWN ) {
                if ( ( _scrollbars[1] && _scrollbars[1]->keyPressed( evt, 0, 0 ) ) ||
                     ( _scrollbars[0] && _scrollbars[0]->keyPressed( evt, 0, 0 ) ) )
                        return true;
        }

        return Widget::keyPressed( evt, x, y );
}

void ScrollpaneWidget::scrollTo( int x, int y ) {
        _scrollViewport->setViewPosition( SCROLL_HORIZONTAL, x );
        _scrollViewport->setViewPosition( SCROLL_VERTICAL, y );
}                
