/*
 * server/LightSource.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "LightSource.h"
#include "../util/Persist.h"

#include <iostream>

int LightSource::_numReserved = 0;

class LightSource LightSource::_lights[ LightSource::MAX_LIGHTS ];

void LightSource::init( float x, float y, float z,
                        unsigned char r,
                        unsigned char g,
                        unsigned char b,
                        unsigned short intensity ) {
        
        _pos[0] = x;
        _pos[1] = y;
        _pos[2] = z;
        _rgb[0] = r;
        _rgb[1] = g;
        _rgb[2] = b;
        _intensity = intensity;
        _sect = 0;
        _delta = 0;
        _die_tick = -1;
        _reserved = false;
}
LightSource::LightSource( float x, float y, float z,
                          unsigned char r,
                          unsigned char g,
                          unsigned char b,
                          unsigned short intensity ) {
        init ( x, y, z, r, g, b, intensity );
}

LightSource::LightSource() {
        init( 0, 0, 0, 0, 0, 0, 0 );

        static int num = 0;
        _num = num++;
}

LightSource::LightSource( class istream & is ) {
        _num = -1;
        _pos[0] = Persist::readShort( is );
        _pos[1] = Persist::readShort( is );
        _pos[2] = Persist::readShort( is );
        
        _rgb[0] = is.get();
        _rgb[1] = is.get();
        _rgb[2] = is.get();

        _intensity = Persist::readShort( is );
        
        //        cout << *this << endl;
}


void LightSource::writeLightSource( class ostream & os ) const {
        Persist::writeShort( os, (short) _pos[0] );
        Persist::writeShort( os, (short) _pos[1] );
        Persist::writeShort( os, (short) _pos[2] );

        os.put( _rgb[0] );
        os.put( _rgb[1] );
        os.put( _rgb[2] );

        Persist::writeShort( os, _intensity );
}

class LightSource * LightSource::getLight( float x, float y, float z,
                                           unsigned char r,
                                           unsigned char g,
                                           unsigned char b,
                                           unsigned short intensity,
                                           short delta,
                                           int die ) {

        if ( _numReserved >= MAX_LIGHTS ) {
                return 0;
        }
        
        for ( int i = 0; i < MAX_LIGHTS; ++i ) {
                if ( _lights[i]._reserved )
                        continue;
                _lights[i].init( x, y, z, r, g, b, intensity );
                _lights[i]._delta = delta;
                _lights[i]._die_tick = die;
                _lights[i]._reserved = true;
                _lights[i]._num = i;
                ++_numReserved;
                return & _lights[i];
        }

        return 0;
}

void LightSource::release() {

        if ( _num >= 0 && _num < MAX_LIGHTS && this == & _lights[ _num ] ) {
                _reserved = false;
                --_numReserved;
        }
}


typedef float * Vec3D;

ostream & operator<<( ostream & os, const Vec3D & v ) {
        os << "(" << v[0] << "," << v[1] << "," << v[2] << ")";
        return os;
}

ostream & operator<<( ostream & os, const LightSource & light ) {
        os << " LightSource: [ " << (Vec3D) light._pos << ", "
           << "(" << (int) light._rgb[0] << ", " << (int) light._rgb[1] << ", " << (int)light._rgb[2] << ")"
           << ", i=" << light._intensity << "]";
        return os;
}
