/*
 * server/Zone_pvs.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include <iostream>
#include <iomanip>
#include <algo.h>

#include "Zone.h"
#include "Sector.h"
#include "LineDef.h"
#include "SideDef.h"
#include "../util/VMath.h"


static void makeSectsVisible( const class Zone * zone,
                              class Sector * s1, 
                              class Sector * s2 ) {

        if ( s1 == s2 )
                return;

        if ( ! s1->canSeeSector( s2 ) )
                s1->_visibleSectors.push_back( s2->_num );
        if ( ! s2->canSeeSector( s1 ) )
                s2->_visibleSectors.push_back( s1->_num );
}

bool Zone::createVisibleSectorSets() {
        _isVisLoaded = true;

        cout << " :: Creating visible  sector sets: [  0%]\b\b";
        
        for ( int i = 0; i < _sectors.length; ++i ) {
                if ( _sectors[i]->_visibleSectors.size() )
                        _sectors[i]->_visibleSectors.erase( _sectors[i]->_visibleSectors.begin(),
                                                            _sectors[i]->_visibleSectors.end() );
        }

        for ( int i = 0; i < _lines.length; ++i ) {
                

                int pct = i * 100 / _lines.length;
                
                cout << "\b\b\b" << setw(3) << pct << flush;

                class LineDef * testLine = _lines[i];
                
                if ( ! testLine->sides[1] )
                        continue;
                
                //                cout << " testline " << *testLine << endl;

                makeSectsVisible( this,
                                  testLine->sides[0]->sector,
                                  testLine->sides[1]->sector );

                float test_dx = testLine->verts[1][0] - testLine->verts[0][0];
                float test_dy = testLine->verts[1][1] - testLine->verts[0][1];

                for ( int j = 0; j < _lines.length; ++j ) {
                        
                        class LineDef * otherLine = _lines[j];
                        
                        if ( otherLine == testLine || ! otherLine->sides[1] )
                                continue;
                        
                        if ( sectorsCanSee( testLine->sides[0]->sector, otherLine->sides[0]->sector ) &&
                             sectorsCanSee( testLine->sides[0]->sector, otherLine->sides[1]->sector ) &&
                             sectorsCanSee( testLine->sides[1]->sector, otherLine->sides[0]->sector ) &&
                             sectorsCanSee( testLine->sides[1]->sector, otherLine->sides[1]->sector ) ) {
                                continue;
                        }

                        float other_dx = otherLine->verts[1][0] - otherLine->verts[0][0];
                        float other_dy = otherLine->verts[1][1] - otherLine->verts[0][1];
                        
                        for ( float t_test = 0.1; t_test < 1; t_test += 0.1 ) {

                                float test_pt[2] = { 
                                        testLine->verts[0][0] + t_test * test_dx,
                                        testLine->verts[0][1] + t_test * test_dy
                                };
                                
                                for ( float t_other = 0.1; t_other < 1; t_other += 0.1 ) {
                                        
                                        float other_pt[2] = { 
                                                otherLine->verts[0][0] + t_other * other_dx,
                                                otherLine->verts[0][1] + t_other * other_dy
                                        };
                                        
                                        bool found = false;
                                        
                                        for ( int k = 0; k < _lines.length; ++k ) {
                                                class LineDef * blocker = _lines[k];

                                                if ( blocker == otherLine || blocker == testLine )
                                                        continue;
                                                
                                                if ( blocker->sides[1] )
                                                        continue;
                                                
                                                float u[2];
                                                
                                                if ( VMath::lineSegmentsIntersect2d( test_pt,
                                                                                       other_pt,
                                                                                       blocker->verts[0],
                                                                                       blocker->verts[1],
                                                                                       &u[0], &u[1] ) ) {
                                                        found = true;
                                                }

                                        }

                                        if ( found == false ) {
                                                /*
                                                cout << " line available from sectors "
                                                     << testLine->sides[0]->sector->_num << ","
                                                     << testLine->sides[1]->sector->_num << " to "
                                                     << otherLine->sides[0]->sector->_num << ","
                                                     << otherLine->sides[1]->sector->_num << endl;
                                                
                                                cout << " from " << test_pt[0] << "," << test_pt[1] << " -> " << other_pt[0] << "," << other_pt[1] << endl;
                                                */
                                                makeSectsVisible( this,
                                                                  testLine->sides[0]->sector,
                                                                  otherLine->sides[0]->sector );
                                                
                                                makeSectsVisible( this,
                                                                  testLine->sides[0]->sector,
                                                                  otherLine->sides[1]->sector );
                                                
                                                makeSectsVisible( this,
                                                                  testLine->sides[1]->sector,
                                                                  otherLine->sides[0]->sector );
                                                
                                                makeSectsVisible( this,
                                                                  testLine->sides[1]->sector,
                                                                  otherLine->sides[1]->sector );
                                                
                                                goto visible_line;
                                        }
                                }
                        }
                visible_line:
                        continue;
                        
                }
        }

        cout << "\b\b\b" << 100 << "%] -- Done." << endl;
        
        return true;
}

/**
 * this method constructs the _visibleFloors and _visibleCeilings vectors for each sector
 * ceilings and floors that are higher than the sector's floor and ceiling, respectively,
 * are culled.
 * the resulting vectors are then sorted according to their heights.
 * the resulting sorted vectors are then subsorted according to matching textures
 */

class Compare_surface_textures {
        const class Array<Sector *> * _mySectors;
        Sector::SurfaceType _which;
public:
        Compare_surface_textures( Sector::SurfaceType which,
                        const class Array<Sector *> * sectors ) {
                _mySectors = sectors;
                _which = which;
        }
        
        int operator() ( const int i, const int j ) const {
                class Sector * sect  = (*_mySectors)[ i ];
                class Sector * vsect = (*_mySectors)[ j ];
                
                if ( _which == Sector::FLOOR ) {
                        
                        return ( vsect->getFloorTex() != sect->getFloorTex() );
                        
                }
                else {
                        
                        return ( vsect->getCeilingTex() != sect->getCeilingTex() );
                        
                }
                
                return 0;
        }
};

class Compare_surfaces {
        const class Array<Sector *> * _mySectors;
        Sector::SurfaceType _which;
public:
        Compare_surfaces( Sector::SurfaceType which,
                        const class Array<Sector *> * sectors ) {
                _mySectors = sectors;
                _which = which;
        }
        
        int operator() ( const int i, const int j ) const {
                class Sector * sect  = (*_mySectors)[ i ];
                class Sector * vsect = (*_mySectors)[ j ];
                
                if ( _which == Sector::FLOOR ) {
                        
                        return ( vsect->getFloorHeight() > sect->getFloorHeight() );
                        
                }
                else {
                        
                        return ( vsect->getCeilingHeight() < sect->getCeilingHeight() );
                        
                }
                
                return 0;
        }
};

bool Zone::createVisibleSurfaceSets() {

        cout << " :: Creating visible surface sets: [   %]\b\b";
        
        for ( int sect_i = 0; sect_i < _sectors.length; ++sect_i ) {
                class Sector * sect = _sectors[ sect_i ];
                
                int pct = sect_i * 100 / _sectors.length;
                
                cout << "\b\b\b" << setw(3) << pct << flush;

                //
                // erase any old vis data
                //

                if ( sect->_visibleFloors.size() )
                        sect->_visibleFloors.erase( sect->_visibleFloors.begin(),
                                                    sect->_visibleFloors.end() );

                if ( sect->_visibleCeilings.size() )
                        sect->_visibleCeilings.erase( sect->_visibleCeilings.begin(),
                                                      sect->_visibleCeilings.end() );

                //
                //
                //

                for ( vector<int>::const_iterator iter = sect->_visibleSectors.begin();
                      iter < sect->_visibleSectors.end(); ++iter ) {
                
                        class Sector * vsect = _sectors[ *iter ];

                        if ( vsect->getFloorHeight() < sect->getCeilingHeight() )
                                sect->_visibleFloors.push_back( *iter );
                        
                        if ( vsect->getCeilingHeight() > sect->getFloorHeight() )
                                sect->_visibleCeilings.push_back( *iter );
                }

                if ( sect->_visibleFloors.size() ) {
                        /*
                        sort( sect->_visibleFloors.begin(), 
                              sect->_visibleFloors.end(),
                              Compare_surface_textures( Sector::FLOOR, &_sectors ) );
                        */
                        stable_sort( sect->_visibleFloors.begin(), 
                                     sect->_visibleFloors.end(),
                                     Compare_surfaces( Sector::FLOOR, &_sectors ) );
                }
                if ( sect->_visibleCeilings.size() ) {
                        /*
                        sort( sect->_visibleCeilings.begin(), 
                              sect->_visibleCeilings.end(),
                              Compare_surface_textures( Sector::CEILING, &_sectors ) );
                        */
                        stable_sort( sect->_visibleCeilings.begin(), 
                                     sect->_visibleCeilings.end(),
                                     Compare_surfaces( Sector::CEILING, &_sectors ) );
                }
        }

        cout << "\b\b\b" << 100 << "%] -- Done." << endl;

        return true;
}

/**
 *
 *
 */

bool Zone::createVisibleLineSets() {

        cout << " :: Creating visible   lines sets: [   %]\b\b";

        for ( int sect_i = 0; sect_i < _sectors.length; ++sect_i ) {
                class Sector * sect = _sectors[ sect_i ];
        
                int pct = sect_i * 100 / _sectors.length;
                
                cout << "\b\b\b" << setw(3) << pct << flush;
                
                //
                // erase any old vis data
                //

                if ( sect->_visibleLines.size() )
                        sect->_visibleLines.erase( sect->_visibleLines.begin(),
                                                   sect->_visibleLines.end() );
                

                for ( vector<int>::const_iterator iter = sect->_visibleSectors.begin();
                      iter < sect->_visibleSectors.end(); ++iter ) {
                
                        class Sector * vsect = _sectors[ *iter ];
                        
                        for ( unsigned int i = 0; i < vsect->_myLines.size(); ++i ) {

                                int vline_i = vsect->_myLines[ i ];
                                
                                //
                                // if we already have this one, skip it
                                //
                                
                                if ( find( sect->_visibleLines.begin(),
                                           sect->_visibleLines.end(),
                                           vline_i ) != sect->_visibleLines.end()  ||
                                     find( sect->_myLines.begin(),
                                           sect->_myLines.end(),
                                           vline_i ) != sect->_myLines.end() ) {
                                        continue;
                                }

                                const class LineDef * line = _lines[ vline_i ];
                                
                                //
                                // front side of line is facing vsect
                                //
                                
                                if ( line->sides[0]->sector == vsect ) {
                                        
                                        //
                                        // ...unless it is 2-sided and no wall subsections exist on the back side
                                        //
                                        
                                        if ( line->sides[1] && 
                                             line->sides[0]->hasWallSection( line->sides[1], 0 ) == false &&
                                             line->sides[0]->hasWallSection( line->sides[1], 2 ) == false )
                                                continue;

                                        for ( int j = 0; j < sect->verts.length; ++j ) {

                                                //
                                                // if any of the sect verts are on the the front side of this line,
                                                // then it cannot be culled...
                                                //

                                                if ( VMath::whichSide( line->verts[0], line->verts[1], sect->verts[j] ) == 0 ) {

                                                        sect->_visibleLines.push_back( vline_i );
                                                        break;
                                                }
                                        }
                                }

                                //
                                // back side of line is facing vsect
                                // TODO: special handling for 2-sided lines?
                                //

                                else {
                                        for ( int j = 0; j < sect->verts.length; ++j ) {

                                                //
                                                // ...unless no wall subsections exist on the front side
                                                //
                                                
                                                if ( line->sides[1]->hasWallSection( line->sides[0], 0 ) == false &&
                                                     line->sides[1]->hasWallSection( line->sides[0], 2 ) == false )
                                                        continue;
                                                

                                                //
                                                // if any of the sect verts are on the the front side of this line,
                                                // then it cannot be culled...
                                                //
                                                
                                                if ( VMath::whichSide( line->verts[1], line->verts[0], sect->verts[j] ) == 0 ) {

                                                        sect->_visibleLines.push_back( vline_i );
                                                        break;
                                                }
                                        }
                                }
                        }
                }
        }

        cout << "\b\b\b" << 100 << "%] -- Done." << endl;

        return true;
}
bool Zone::createVisibleSets() {
        return createVisibleSectorSets() && 
                createVisibleSurfaceSets() && 
                createVisibleLineSets();
}


bool Zone::mergePVS( class Zone & other ) {

        if ( _sectors.length != other._sectors.length ) {
                cout << " _sectors.length= " << _sectors.length 
                     << " but other._sectors.length= " << other._sectors.length 
                     << endl;
                return false;
        }
        
        for ( int sect_i = 0; sect_i < _sectors.length; ++sect_i ) {
                
                class Sector * sect = _sectors[ sect_i ];
                class Sector * other_sect = other._sectors[ sect_i ];

                //
                // erase any old vis data
                //

                if ( sect->_visibleSectors.size() )
                        sect->_visibleSectors.erase( sect->_visibleSectors.begin(),
                                                     sect->_visibleSectors.end() );
                
                if ( sect->_visibleFloors.size() )
                        sect->_visibleFloors.erase( sect->_visibleFloors.begin(),
                                                    sect->_visibleFloors.end() );

                if ( sect->_visibleCeilings.size() )
                        sect->_visibleCeilings.erase( sect->_visibleCeilings.begin(),
                                                      sect->_visibleCeilings.end() );
                if ( sect->_visibleLines.size() )
                        sect->_visibleLines.erase( sect->_visibleLines.begin(),
                                                   sect->_visibleLines.end() );        

                
                sect->_visibleSectors.insert( sect->_visibleSectors.begin(),
                                              other_sect->_visibleSectors.begin(),
                                              other_sect->_visibleSectors.end() );

                sect->_visibleFloors.insert( sect->_visibleFloors.begin(),
                                             other_sect->_visibleFloors.begin(),
                                             other_sect->_visibleFloors.end() );

                sect->_visibleCeilings.insert( sect->_visibleCeilings.begin(),
                                               other_sect->_visibleCeilings.begin(),
                                               other_sect->_visibleCeilings.end() );

                sect->_visibleLines.insert( sect->_visibleLines.begin(),
                                            other_sect->_visibleLines.begin(),
                                            other_sect->_visibleLines.end() );

                _isVisLoaded = other._isVisLoaded;
                
        }

        
        return true;
}
