/*
 * widgets/Viewport.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "Viewport.h"

Viewport::Viewport( int w, int h, class Widget * view ) 
        : Widget( w, h )
{
        _name = "Viewport";
        add( view, 0, 0 );
        _view = view;
        const class Viewport * old_v = view->getViewport();
                
        if ( old_v ) {
                cout << " :: error, viewport is superseding existing viewport "
                     << *old_v << " on view " << *view << endl;
        }
        _view->setViewport( this );

        _viewPosition[0] = 0;
        _viewPosition[1] = 0;

        setDrawMargin( false );
}

Viewport::~Viewport() {
        if ( _view )
                _view->setViewport( 0 );
        _view = 0;
}

void Viewport::paint() {
        glColor3f( 1, 1, 1 );
        glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
        glBegin( GL_QUADS );
        {
                glVertex2d( 0,            0 );
                glVertex2d( _size[0] - 1, 0 );
                glVertex2d( _size[0] - 1, _size[1] - 1 );
                glVertex2d( 0,            _size[1] - 1 );
        }
        glEnd();

        glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

        return;
}


void Viewport::display() {

        glPushMatrix();
        {

                glTranslatef( _pos[0], _pos[1], 0 );
                
                glScissor( _absolutePos[0], _absolutePos[1], _size[0], _size[1] );
                glEnable( GL_SCISSOR_TEST );

                _view->display();

                glDisable( GL_SCISSOR_TEST );

                glDisable( GL_CLIP_PLANE0 );
                glDisable( GL_CLIP_PLANE1 );
                glDisable( GL_CLIP_PLANE2 );
                glDisable( GL_CLIP_PLANE3 );
                
                paint();
                        
        }
        glPopMatrix();
}
        

void Viewport::setViewPosition( enum ScrollType type, int amount ) {
        if ( _view == 0 )
                return;
        int x = _viewPosition[ 0 ], y = _viewPosition[ 1 ];
                
        _viewPosition[ type ] = amount - ( amount % _view->getScrollableUnitIncrement( type ) );
        _viewPosition[ type ] = min( max( 0, _viewPosition[type]  ), 
                                     _view->getSize()[type] - _size[type] );
        _view->setPosition( -_viewPosition[0], -_viewPosition[1] );
                
        notifyViewportListeners( this, x, y );

}

void Viewport::scrollUnits( enum ScrollType type, int amount ) {
        int x = _viewPosition[ 0 ], y = _viewPosition[ 1 ];
                
        _viewPosition[ type ] += amount * _view->getScrollableUnitIncrement( type );
                
        _viewPosition[ type ] = min( max( 0, _viewPosition[type]  ), 
                                     _view->getSize()[type] - _size[type] );
        _view->setPosition( -_viewPosition[0], -_viewPosition[1] );

        notifyViewportListeners( this, x, y );
}

void Viewport::scrollBlocks( enum ScrollType type, int amount ) {
        int x = _viewPosition[ 0 ], y = _viewPosition[ 1 ];
        _viewPosition[ type ] += amount * _view->getScrollableBlockIncrement( type );
        _viewPosition[ type ] = min( max( 0, _viewPosition[type]  ), 
                                     _view->getSize()[type] - _size[type] );
        _view->setPosition( -_viewPosition[0], -_viewPosition[1] );
        notifyViewportListeners( this, x, y );
}

/**
         * returns the new  position of the viewport along the specified axis
         */
int Viewport::scrollPixels( enum ScrollType type, int amount ) {
        int x = _viewPosition[ 0 ], y = _viewPosition[ 1 ];
        amount /= _view->getScrollableUnitIncrement( type );

        if ( amount ) {
                scrollUnits( type, amount );
                notifyViewportListeners( this, x, y );
        }
        return _viewPosition[ type ];
}
