/*
 * widgets/WidgetEvents.h:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef __WidgetEvents_h__
#define __WidgetEvents_h__

/**
 * base class for Events in the Observer-Producer / Listener-Producer model
 */

enum WidgetEventType {
        EVENT_NONE     = 0,
        EVENT_ACTION   = 1,
        EVENT_RESIZE   = 2,
        EVENT_VIEWPORT = 3,
        EVENT_NUM      = 4
};

class WidgetEvent {
protected:
        const class Widget * _source;
        
        WidgetEvent( const class Widget * source, enum WidgetEventType type ) {
                _type   = type;
                _source = source;
        }
        
        enum WidgetEventType _type;
public:
        
        const enum WidgetEventType getType() const { return _type; }
        const class Widget * getSource() const { return _source; }
        
        friend class ostream & operator<<( class ostream & os, const class WidgetEvent & evt );
};

/**
 * an event which represents a binary or atomic action
 */
class ActionEvent : public WidgetEvent {
protected:
        char *_actionCommand;
public:

        ActionEvent( const class Widget * source, const char * command );
        ~ActionEvent();

        const char * getActionCommand() const { return _actionCommand; }

};

class ResizeEvent : public WidgetEvent {
        int _oldSize[2];
        bool _sizeChanged[2];
public:
        ResizeEvent( const class Widget * source, int old_x, int old_y );
        
        const int * getOldSize() const { return _oldSize; }
        const bool getSizeChanged( int type ) const { return _sizeChanged[ type ]; }
};


class ViewportEvent : public WidgetEvent {
protected:
        int _newViewPosition[2];
        int _oldViewPosition[2];
        bool _orientation[2];
public:
        
        ViewportEvent( const class Viewport * source, 
                       int old_x, 
                       int old_y, 
                       int new_x, 
                       int new_y ) : WidgetEvent( (Widget * ) source, EVENT_VIEWPORT ) {
                
                _newViewPosition[0] = new_x;
                _newViewPosition[1] = new_y;

                _oldViewPosition[0] = old_x;
                _oldViewPosition[1] = old_y;

                _orientation[ 0 ] = _oldViewPosition[ 0 ] != _newViewPosition[ 0 ];
                _orientation[ 1 ] = _oldViewPosition[ 1 ] != _newViewPosition[ 1 ];
        }
        
        const int * getOldPos() const { return _oldViewPosition; }
        const int * getNewPos() const { return _newViewPosition; }
        const bool orientationChanged( int type ) const { return _orientation[ type ]; }
};


//
//
// LISTENER UTILITY SUBCLASSES
//
//

/**
 * a listener for WidgetEvents
 */
class WidgetEventListener {
        bool _acceptedTypes[EVENT_NUM];
        bool _enabled;
        void * ptr;
protected:
        void acceptEventType( enum WidgetEventType type, bool flag ) {
                _acceptedTypes[type] = flag;
        }

        void setListenerEnabled( bool flag ) {
                _enabled = flag;
        }
public:

        virtual void handleEvent( const class WidgetEvent *evt ) = 0;
        
        bool eventTypeOk( enum WidgetEventType type ) const;

        WidgetEventListener();

        friend class ostream & operator<<( class ostream & os, const class WidgetEventListener & listener );
};


class ViewportListener : public WidgetEventListener {
public:
        ViewportListener() : WidgetEventListener() {
                acceptEventType( EVENT_VIEWPORT, true );
        }
};
  
#endif
