/*
 * util/BitmapFont.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "GFX.h"
#include "BitmapFont.h"
#include "Exception.h"
#include "Texload.h"

#include <iostream>
#include <iomanip>
#include <string>
#include <algorithm>
#include <cstdio>
#include <fstream>

const string BitmapFont::NAME = "BitmapFont";

/**
 * PC Screen Font
 * from redhat linux package console-tools
 * URL         : http://www.multimania.com/ydirson/en/lct/
 * file format
 */

GLubyte * loadPSFBitmapFont( int count, const char *fontname, int *height ) {

        static int num_chars_per_filemode[4] = { 
                256, 512, 256, 512
        };

        glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );        
        glPixelStorei( GL_UNPACK_ROW_LENGTH, 8 );

        char filename[ 1024 ];
        sprintf( filename, "%s/psf/%s.psf", GFX::BITMAP_FONT_DIR.c_str(), fontname );

        ifstream ifs( filename );
        
        if ( ! ifs ) {
                cout << " :: unable to open " << filename << endl;
                return 0;
        }

        unsigned char magic_number[2];

        ifs.read( magic_number, 2 );

        if (  magic_number[0] != 0x36 || magic_number[1] != 0x04 ) {
                cout << " file " << filename << " does not have the PSF magic number 0x0436" << endl;
                return 0;
        }

        unsigned char filemode = ifs.get();
        unsigned char fontheight = ifs.get();

        cout << " :: psf font height: " << (int)fontheight << ", filemode: " << (int)filemode << endl;
        
        
        count = min( count, num_chars_per_filemode[ (int) filemode ] );

        *height = fontheight;

        GLubyte * bitmap = new GLubyte[ count * fontheight ];

        ifs.read( bitmap, count * fontheight );
 
        if ( ifs.gcount() != (unsigned int) ( count * fontheight ) ) {
                cout << " :: only read " << ifs.gcount() << ", not " << count * fontheight << endl;
                delete[] bitmap;
                return 0;
        }
        
        //
        // glbitmap fonts first scanline is the bottom, not the top.
        // PSF fonts first scanline is the top, so we must flip all the scanlines
        //
        
        GLubyte *ptr = bitmap;
        for ( int i = 0; i < count; ++i ) {
                for ( int j = 0; j < ( fontheight / 2 ); ++j ) {                
                        GLubyte tmp                   = *( ptr + j );
                        *( ptr + j )                  = *( ptr + fontheight - j - 1 );
                        *( ptr + fontheight - j - 1 ) = tmp;
                }
                ptr += fontheight;
        }
        return bitmap;
}
                
        
        
GLubyte * loadHexBitmapFont( int start, int count, const char *fontname, int *height ) {

        glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
        
        glPixelStorei( GL_UNPACK_ROW_LENGTH, 8 );

        char filename[ 1024 ];
        sprintf( filename, "%s/hex/%s.hex", GFX::BITMAP_FONT_DIR.c_str(), fontname );

        ifstream ifs( filename );
    
        if ( ! ifs ) {
                cout << " unable to open " << filename << endl;
                return 0;
        }    

        char linebuf[1024];
        char hexstr[16];

        strcpy( hexstr, "0x" );
        hexstr[6] = 0;

        GLubyte * bitmap = new GLubyte[ count * 16 ];

        while ( ifs.getline( linebuf, 1024 ) ) {
                strncpy( hexstr+2, linebuf, 4 );
                int num = strtol( hexstr, 0, 16 );
                        
                       
                if ( num < start || num >= start + count ) {
                        continue;
                }
                        
                num -= start;

                hexstr[4] = 0;

                for ( int i = 5, j = 15; i < 37; i+=2, --j ) {
                        strncpy( hexstr + 2, linebuf + i, 2 );
                        GLubyte smallnum = strtol( hexstr, 0, 16 );
                        bitmap[ num*16 + j ] = smallnum;
                }
                                
        }

        ifs.close();

        *height = 16;

        return bitmap;
}

BitmapFont::BitmapFont( const char * const fontName ) throw ( class Exception ) : AbstractFont( fontName ) {
        
        _bitmap = 0;

        cout << " :: BitmapFont loading " << _fontName << endl;

        if ( ! initGlyphLists( MAX_GLYPHS ) ) {
                throw Exception( "BitmapFont unable to allocate display lists." );
        }

        //        _bitmap = loadHexBitmapFont( 0, 256, fontName, &_height );
        _bitmap = loadPSFBitmapFont( 256, fontName, &_height );

        if ( ! _bitmap ) {
                cout << " unable to load font bitmap for " << fontName  << endl;
                throw Exception( " unable to load font bitmap for " );
        }
        
        for ( int i = 0; i < MAX_GLYPHS; ++i ) {
                _leads[ i ]  = 0;
                _widths[ i ] = 9;
                glNewList( i + _glyphsIndex, GL_COMPILE );
                {
                        glBitmap( 8, _height, 0, 0, 9, 0, _bitmap + (i*_height) );
                }
                glEndList();
        }
}

BitmapFont::~BitmapFont() {
        if ( _bitmap )
                delete _bitmap;
}

void BitmapFont::startStringDraw() const {
        glListBase( _glyphsIndex );

}

void BitmapFont::endStringDraw() const {
}
