/*
 * widgets/TableWidget.cc:
 *
 * Copyright (C) 2000 John Watson, jwatson@tempusmud.com
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#include "TableWidget.h"
#include "AbstractTableModel.h"

#include "LabelWidget.h"
#include "../util/TextureFont.h"

TableWidget::TableWidget( int w, int h, 
                          class AbstractTableModel * model )
        : Widget( w, h )
{
        _header = 0;
        _model = model;

        _rowHeight = 20;
        _name = "TableWidget";
        int numCols = _model->getColumnCount();

        _colWidths = new int[ numCols ];
        
        for ( int i = 0; i < numCols; ++i ) {
                _colWidths[i] = w / numCols;
        }

        setFGColor4fv( getLookAndFeel()->getBlackColor() );
}

/**
 * the Table is not responsible for deleting the TableModel
 * the Table does, however, DELETES THE HEADER, so beware
 * also note that the TableModel MUST NOT BE DELETED before
 * the table or the header
 */
TableWidget::~TableWidget() {
        delete[] _colWidths;
        if ( _header )
                delete _header;
        //
        // don't delete the model
        //
}

void TableWidget::paint() {
        Widget::paint();

        int gridsize[2] = { _model->getColumnCount(),
                            _model->getRowCount() };

        const class Viewport * v = 0;

        int clip[4];
        
        int startrow;
        int endrow;

        if ( ( v = findViewportClipRectangle( clip ) ) ) {
                startrow  = getSize()[1] - clip[1] - clip[3];
                endrow    = startrow + clip[3];

        }
        else {
                startrow  = 0;
                endrow    = getSize()[1];
        }
        
        startrow /= _rowHeight;
        endrow   /= _rowHeight;
        
        startrow = max( 0, startrow );
        endrow   = min( gridsize[1] - 1, endrow );

        //
        // number of horizontal cell lines: numRows - 1
        // number of vertical cell lines: numCols - 1
        //
        
        int numHorizLines = endrow - startrow;
        
        int numVertLines  = gridsize[0] ? ( gridsize[0] - 1 ) : 0;
        
        if ( numVertLines || numHorizLines ) {
                
                const int MAX_LINES = 50;
                numHorizLines = min( MAX_LINES, numHorizLines );
                numVertLines = min( MAX_LINES - numHorizLines, numVertLines );
                        
                int lines[ MAX_LINES /*numVertLines + numHorizLines*/ ][2][2];
                
                int index = 0;
                
                for ( int i = startrow + 1; i <= endrow; ++i, ++index ) {

                        lines[ index ][0][0] = 0;
                        lines[ index ][0][1] = _size[1] - i * _rowHeight;
                        lines[ index ][1][0] = _size[0];
                        lines[ index ][1][1] = lines[ index ][0][1];
                }

                int cur_x = 0;
                for ( int i = 1; i < gridsize[0]; ++i, ++index ) {
                        cur_x += getColumnWidth( i-1 );
                        lines[ index ][0][0] = cur_x;
                        lines[ index ][0][1] = 0;
                        lines[ index ][1][0] = cur_x;
                        lines[ index ][1][1] = _size[1];
                }

                glColor4fv( getFGColor4fv() );

                glBegin( GL_LINES );
                {
                        
                        int tot_lines = numHorizLines + numVertLines;

                        for ( int i = 0; i < tot_lines; ++i ) {
                                glVertex2iv( lines[ i ][0] );
                                glVertex2iv( lines[ i ][1] );
                        }
                }
                glEnd();

        }

        int pos[2] = { 0, _size[1] };
        glColor4fv( getFGColor4fv() );

        _font->startStringDraw();
        for ( int y = 0; y < gridsize[1]; ++y ) {
                pos[0] = 0;
                pos[1] -= _rowHeight;

                if ( y < startrow ) {
                        continue;
                }

                if ( y > endrow ) {
                        break;
                }

                for ( int x = 0; x < gridsize[0]; ++x ) {
                        TableCellType type = _model->getColumnType( x );
                        if ( type == TABLE_CELL_CHAR_PTR ) {
                                glPushMatrix();
                                glTranslatef( pos[0], pos[1], 0 );
                                const char * str = (const char *) _model->getValueAt( y, x );
                                _font->drawStringFragment( str, 0, strlen( str ) );
                                glPopMatrix();

                        }

                        pos[0] += getColumnWidth( x );
                }
        }
        _font->endStringDraw();
}


void TableWidget::setColumnWidths( const int * widths ) {
        int numcol = _model->getColumnCount();
        _colWidths = new int[ numcol ];
        
        for ( int i = 0; i < numcol; ++i ) {
                _colWidths[ i ] = widths[ i ];
        }
}

class TableHeader * TableWidget::getHeader() {
        if ( _header == 0 ) {
                _header = new TableHeader( this );
        }
        return _header;
}

TableHeader::TableHeader( class TableWidget * table ) 
        : Widget( table->getSize()[0], 20 ),
        _table( table )
{
        int numcols = _table->getModel()->getColumnCount();
        
        _labels = new LabelWidget * [ numcols ];

        int cur_x = 0;

        for ( int i = 0; i < numcols; ++i ) {
                const char * str = _table->getModel()->getColumnName( i );
                _labels[i] = new LabelWidget( _table->getColumnWidth( i ), 
                                              _size[1],
                                              str ? str : "ERROR" );
                add( _labels[i], cur_x, 0 );
                cur_x += _labels[i]->getSize()[0];
        }
}

TableHeader::~TableHeader() {
        int numCols = _table->getModel()->getColumnCount();

        for ( int i = 0; i < numCols; ++i ) {
                delete _labels[i];
        }
        
        delete[] _labels;
}
